"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const events = require("@aws-cdk/aws-events");
const defaults = require("@aws-solutions-constructs/core");
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventsRuleToSns(stack, 'test', props);
}
test('snapshot test EventsRuleToSns default params', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check if the event rule has permission/policy in place in sns for it to be able to publish to the topic', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SNS::TopicPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        StringEquals: {
                            "AWS:SourceOwner": {
                                Ref: "AWS::AccountId"
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "TopicOwnerOnlyAccess"
                },
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: "*",
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "HttpsOnly"
                },
                {
                    Action: "sns:Publish",
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "2"
                }
            ],
            Version: "2012-10-17"
        },
        Topics: [
            {
                Ref: "testSnsTopic42942701"
            }
        ]
    });
});
test('check events rule properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "testSnsTopic42942701"
                },
                Id: {
                    "Fn::GetAtt": [
                        "testSnsTopic42942701",
                        "TopicName"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    expect(construct.eventsRule !== null);
    expect(construct.snsTopic !== null);
    expect(construct.encryptionKey !== null);
});
test('check the sns topic properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testEncryptionKeyB55BFDBC",
                "Arn"
            ]
        }
    });
});
test('check the sns topic properties with existing KMS key', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKey: key
    };
    new lib_1.EventsRuleToSns(stack, 'test-events-rule-sqs', props);
    expect(stack).toHaveResource('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    expect(stack).toHaveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        "kms:Create*",
                        "kms:Describe*",
                        "kms:Enable*",
                        "kms:List*",
                        "kms:Put*",
                        "kms:Update*",
                        "kms:Revoke*",
                        "kms:Disable*",
                        "kms:Get*",
                        "kms:Delete*",
                        "kms:ScheduleKeyDeletion",
                        "kms:CancelKeyDeletion",
                        "kms:GenerateDataKey",
                        "kms:TagResource",
                        "kms:UntagResource"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:Decrypt",
                        "kms:Encrypt",
                        "kms:ReEncrypt*",
                        "kms:GenerateDataKey*"
                    ],
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: "*"
                }
            ],
            Version: "2012-10-17"
        },
        Description: "my-key",
        EnableKeyRotation: true
    });
});
//# sourceMappingURL=data:application/json;base64,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