"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.STORAGE_KEY_FORMAT_REGEX = exports.CORRUPT_ASSEMBLY_REPORT_PATTERN = exports.MISSING_DOCUMENTATION_REPORT_PATTERN = exports.UNINSTALLABLE_PACKAGES_REPORT = exports.corruptAssemblyReport = exports.missingDocumentationReport = exports.CATALOG_KEY = exports.UNPROCESSABLE_PACKAGE_ERROR_NAME = exports.UNINSTALLABLE_PACKAGE_SUFFIX = exports.CORRUPT_ASSEMBLY_SUFFIX = exports.NOT_SUPPORTED_SUFFIX = exports.notSupportedKeySuffix = exports.corruptAssemblyKeySuffix = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_CSHARP = exports.DOCS_KEY_SUFFIX_JAVA = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_JAVA = docsKeySuffix(language_1.DocumentationLanguage.JAVA);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_CSHARP = docsKeySuffix(language_1.DocumentationLanguage.CSHARP);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * The key suffix for a corrupted assembly marker by language and submodule.
 */
function corruptAssemblyKeySuffix(lang, submodule) {
    return `${docsKeySuffix(lang, submodule)}${exports.CORRUPT_ASSEMBLY_SUFFIX}`;
}
exports.corruptAssemblyKeySuffix = corruptAssemblyKeySuffix;
/**
 * The key suffix for a not supported marker by language and submodule.
 */
function notSupportedKeySuffix(lang, submodule) {
    return `${docsKeySuffix(lang, submodule)}${exports.NOT_SUPPORTED_SUFFIX}`;
}
exports.notSupportedKeySuffix = notSupportedKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * Key suffix for beacon files marking that a language specific assembly is corrupt
 * and we cannot generate docs from it.
 */
exports.CORRUPT_ASSEMBLY_SUFFIX = '.corruptassembly';
/**
 * Key suffix for a beacon file when a package cannot be installed.
 */
exports.UNINSTALLABLE_PACKAGE_SUFFIX = '/uninstallable';
/**
 * Name of the error denoting an unprocessable package that should be diverted away from the DLQ.
 */
exports.UNPROCESSABLE_PACKAGE_ERROR_NAME = 'UnprocessablePackageError';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * The key for missing documentation report.
 *
 * @param language the language for which missing documentation is requested.
 */
function missingDocumentationReport(language) {
    return `missing-objects/${language.name}-documentation.json`;
}
exports.missingDocumentationReport = missingDocumentationReport;
/**
 * The key for corrupt assembly report.
 *
 * @param language the language for which the report is requested.
 */
function corruptAssemblyReport(language) {
    return `corruptassembly-objects/${language.name}.json`;
}
exports.corruptAssemblyReport = corruptAssemblyReport;
/**
 * The key for uninstallable packages report.
 */
exports.UNINSTALLABLE_PACKAGES_REPORT = 'uninstallable-objects/data.json';
/**
 * The key pattern for objects containing missing documentation lists.
 */
exports.MISSING_DOCUMENTATION_REPORT_PATTERN = 'missing-objects/*-documentation.json';
/**
 * The key pattern for objects containing unprocessable assembly lists.
 */
exports.CORRUPT_ASSEMBLY_REPORT_PATTERN = 'corruptassembly-objects/*.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
//# sourceMappingURL=data:application/json;base64,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