"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToStepFunction = void 0;
const defaults = require("@aws-solutions-constructs/core");
const aws_events_rule_step_function_1 = require("@aws-solutions-constructs/aws-events-rule-step-function");
const core_1 = require("@aws-cdk/core");
const cloudtrail = require("@aws-cdk/aws-cloudtrail");
class S3ToStepFunction extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the S3ToStepFunction class.
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        if (!props.existingBucketObj) {
            [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
                bucketProps: props.bucketProps
            });
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        if (!props.hasOwnProperty('deployCloudTrail') || props.deployCloudTrail === true) {
            [this.cloudtrailBucket, this.cloudtrailLoggingBucket] = defaults.buildS3Bucket(this, {}, 'CloudTrail');
            this.cloudtrail = new cloudtrail.Trail(this, 'S3EventsTrail', {
                bucket: this.cloudtrailBucket
            });
            this.cloudtrail.addS3EventSelector([{
                    bucket
                }], {
                readWriteType: cloudtrail.ReadWriteType.ALL,
                includeManagementEvents: false
            });
        }
        let _eventRuleProps = {};
        if (props.eventRuleProps) {
            _eventRuleProps = props.eventRuleProps;
        }
        else {
            // By default the CW Events Rule will filter any 's3:PutObject' events for the S3 Bucket
            _eventRuleProps = {
                eventPattern: {
                    source: ['aws.s3'],
                    detailType: ['AWS API Call via CloudTrail'],
                    detail: {
                        eventSource: [
                            "s3.amazonaws.com"
                        ],
                        eventName: [
                            "PutObject"
                        ],
                        requestParameters: {
                            bucketName: [
                                bucket.bucketName
                            ]
                        }
                    }
                }
            };
        }
        const eventsRuleToStepFunction = new aws_events_rule_step_function_1.EventsRuleToStepFunction(this, 'test-events-rule-step-function-stack', {
            stateMachineProps: props.stateMachineProps,
            eventRuleProps: _eventRuleProps
        });
        this.stateMachine = eventsRuleToStepFunction.stateMachine;
        this.stateMachineLogGroup = eventsRuleToStepFunction.stateMachineLogGroup;
        this.cloudwatchAlarms = eventsRuleToStepFunction.cloudwatchAlarms;
    }
}
exports.S3ToStepFunction = S3ToStepFunction;
//# sourceMappingURL=data:application/json;base64,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