"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DjangoEcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const patterns = require("@aws-cdk/aws-ecs-patterns");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const route53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const database_1 = require("./common/database");
const elasticache_1 = require("./common/elasticache");
const vpc_1 = require("./common/vpc");
const beat_1 = require("./ecs/celery/beat");
const worker_1 = require("./ecs/celery/worker");
const tasks_1 = require("./ecs/tasks");
/**
 * Configures a Django project using ECS Fargate.
 *
 * @stability stable
 */
class DjangoEcs extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        /**
         * VPC must have public, private and isolated subnets
         *
         * If you don't provide a VPC, a new VPC will be created
         */
        if (!props.vpc) {
            const applicationVpc = new vpc_1.ApplicationVpc(scope, 'AppVpc');
            this.vpc = applicationVpc.vpc;
        }
        else {
            const vpc = props.vpc;
            this.vpc = vpc;
        }
        /**
         * static files bucket name is derived from the Construct id if not provided
         */
        const staticFilesBucket = new s3.Bucket(scope, 'StaticBucket', {
            bucketName: props === null || props === void 0 ? void 0 : props.bucketName,
        });
        this.staticFileBucket = staticFilesBucket;
        /**
         * ECS cluster
         */
        this.cluster = new ecs.Cluster(scope, 'EcsCluster', { vpc: this.vpc });
        /**
         * task definition construct
         */
        const taskDefinition = new ecs.TaskDefinition(scope, 'TaskDefinition', {
            compatibility: ecs.Compatibility.FARGATE,
            cpu: '256',
            memoryMiB: '512',
        });
        /**
         * Container image used in web API, celery worker and management task containers
         */
        this.image = new ecs.AssetImage(props.imageDirectory);
        /**
         * RDS managed database using PostgreSQL
         */
        const database = new database_1.RdsPostgresInstance(scope, 'RdsPostgresInstance', {
            vpc: this.vpc,
            dbSecretName: 'dbSecret',
        });
        /**
         * A security group in the VPC for our application (ECS Fargate services and tasks)
         * Allow the application services to access the RDS security group
         */
        const appSecurityGroup = new ec2.SecurityGroup(scope, 'appSecurityGroup', {
            vpc: this.vpc,
        });
        /**
         * ElastiCache Redis Cluster for caching, celery message brokering
         */
        const elastiCacheRedis = new elasticache_1.ElastiCacheCluster(scope, 'ElastiCacheCluster', {
            vpc: this.vpc,
            appSecurityGroup,
        });
        const environment = {
            AWS_STORAGE_BUCKET_NAME: staticFilesBucket.bucketName,
            POSTGRES_SERVICE_HOST: database.rdsPostgresInstance.dbInstanceEndpointAddress,
            DB_SECRET_NAME: database.dbSecretName,
            DEBUG: '0',
            DJANGO_SETTINGS_MODULE: 'backend.settings.production',
            REDIS_SERVICE_HOST: elastiCacheRedis.elastiCacheCluster.attrRedisEndpointAddress,
        };
        taskDefinition.addContainer('backendContainer', {
            image: this.image,
            environment,
            command: props.webCommand,
            portMappings: [{
                    containerPort: 8000,
                    hostPort: 8000,
                }],
            logging: ecs.LogDriver.awsLogs({
                logRetention: logs.RetentionDays.ONE_DAY,
                streamPrefix: 'BackendContainer',
            }),
        });
        new tasks_1.managementCommandTask(scope, 'migrate', {
            image: this.image,
            command: ['python3', 'manage.py', 'migrate', '--no-input'],
            appSecurityGroup,
            environment,
            dbSecret: database.secret,
            cluster: this.cluster,
            // this command will run automatically on every deployment
            run: true,
            vpc: this.vpc,
        });
        new tasks_1.managementCommandTask(scope, 'collectstatic', {
            image: this.image,
            command: ['python3', 'manage.py', 'collectstatic', '--no-input'],
            appSecurityGroup,
            environment,
            dbSecret: database.secret,
            cluster: this.cluster,
            vpc: this.vpc,
        });
        /**
         * Use celery beat if it is configured in props
         */
        if ((_b = props.useCeleryBeat) !== null && _b !== void 0 ? _b : false) {
            new beat_1.CeleryBeat(scope, 'CeleryBeat', {
                dbSecret: database.secret,
                image: this.image,
                command: [
                    'celery',
                    '--app=backend.celery_app:app',
                    'beat',
                    '--loglevel=INFO',
                    '--pidfile=/code/celerybeat.pid',
                ],
                environment,
                cluster: this.cluster,
                securityGroups: [appSecurityGroup],
            });
        }
        ;
        /**
         * Celery worker
         *
         * TODO: refactor to support defining multiple queues
         */
        new worker_1.CeleryWorker(scope, 'CeleryWorkerDefaultQueue', {
            image: this.image,
            command: [
                'celery',
                '-A',
                'backend.celery_app:app',
                'worker',
                '-l',
                'info',
                '-Q',
                'celery',
                '-n',
                'worker-celery@%h',
            ],
            environment,
            cluster: this.cluster,
            securityGroups: [appSecurityGroup],
            dbSecret: database.secret,
        });
        /**
         * Lookup Certificate from ARN or generate
         * Deploy external-dns and related IAM resource if a domain name is included
         */
        let certificate = undefined;
        let hostedZone = undefined;
        if (props.domainName) {
            hostedZone = route53.HostedZone.fromLookup(scope, 'hosted-zone', {
                domainName: props.domainName,
            });
            /**
             * Lookup or request ACM certificate depending on value of certificateArn
             */
            if (props.certificateArn) {
                // lookup ACM certificate from ACM certificate ARN
                certificate = acm.Certificate.fromCertificateArn(scope, 'certificate', props.certificateArn);
            }
            else {
                // request a new certificate
                certificate = new acm.Certificate(this, 'SSLCertificate', {
                    domainName: props.domainName,
                    validation: acm.CertificateValidation.fromDns(hostedZone),
                });
            }
        }
        /**
          * ECS load-balanced fargate service
         */
        const albfs = new patterns.ApplicationLoadBalancedFargateService(scope, 'AlbFargateService', {
            cluster: this.cluster,
            taskDefinition,
            securityGroups: [appSecurityGroup],
            desiredCount: 1,
            assignPublicIp: true,
            redirectHTTP: true,
            certificate: props.domainName ? certificate : undefined,
        });
        database.secret.grantRead(albfs.taskDefinition.taskRole);
        const albLogsBucket = new s3.Bucket(scope, `${id}-alb-logs`);
        albfs.loadBalancer.logAccessLogs(albLogsBucket);
        /**
       * Health check for the application load balancer
       */
        albfs.targetGroup.configureHealthCheck({
            path: '/api/health-check/',
        });
        /**
         * Allows the app security group to communicate with the RDS security group
         */
        database.rdsSecurityGroup.addIngressRule(appSecurityGroup, ec2.Port.tcp(5432));
        /**
         * Grant the task defintion read-write access to static files bucket
         */
        staticFilesBucket.grantReadWrite(albfs.taskDefinition.taskRole);
        new cdk.CfnOutput(this, 'bucketName', { value: staticFilesBucket.bucketName });
        new cdk.CfnOutput(this, 'apiUrl', { value: albfs.loadBalancer.loadBalancerFullName });
        if (props.domainName) {
            new route53.ARecord(scope, 'ARecord', {
                target: route53.RecordTarget.fromAlias(new route53targets.LoadBalancerTarget(albfs.loadBalancer)),
                zone: hostedZone,
                recordName: props.domainName,
            });
        }
    }
}
exports.DjangoEcs = DjangoEcs;
_a = JSII_RTTI_SYMBOL_1;
DjangoEcs[_a] = { fqn: "django-cdk.DjangoEcs", version: "0.0.19" };
//# sourceMappingURL=data:application/json;base64,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