"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DjangoEks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecrAssets = require("@aws-cdk/aws-ecr-assets");
const eks = require("@aws-cdk/aws-eks");
// import * as logs from '@aws-cdk/aws-logs';
const iam = require("@aws-cdk/aws-iam");
const route53 = require("@aws-cdk/aws-route53");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const database_1 = require("./common/database");
const vpc_1 = require("./common/vpc");
const awslbc_1 = require("./eks/awslbc");
// import { ElastiCacheCluster } from './elasticache';
const external_dns_1 = require("./eks/external-dns");
const irsa_1 = require("./eks/irsa");
const ingress_1 = require("./eks/resources/ingress");
const migrate_1 = require("./eks/resources/migrate");
const web_1 = require("./eks/resources/web");
/**
 * Configures a Django project using EKS.
 *
 * @stability stable
 */
class DjangoEks extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        /**
         * VPC must have public, private and isolated subnets
         *
         * If you don't provide a VPC, a new VPC will be created
         */
        if (!props.vpc) {
            const applicationVpc = new vpc_1.ApplicationVpc(scope, 'AppVpc');
            this.vpc = applicationVpc.vpc;
        }
        else {
            this.vpc = props.vpc;
        }
        /**
         * static files bucket name is derived from the Construct id if not provided
         */
        const staticFilesBucket = new s3.Bucket(scope, 'StaticBucket', {
            bucketName: props.bucketName,
            autoDeleteObjects: true,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.staticFileBucket = staticFilesBucket;
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        /**
         * EKS cluster
         */
        this.cluster = new eks.Cluster(this, 'MyEksCluster', {
            version: eks.KubernetesVersion.V1_19,
            vpc: this.vpc,
            mastersRole,
            defaultCapacity: 2,
        });
        /**
         * Namespace for application
         */
        const appNamespace = this.cluster.addManifest('app-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'app',
            },
        });
        /**
         * Creates an IAM role with a trust relationship that is scoped to the cluster's OIDC provider
         * https://docs.aws.amazon.com/eks/latest/userguide/iam-roles-for-service-accounts-technical-overview.html
         *
         * The role (podRole) will be given access the Secrets Manager Secret, the S3 bucket and
         * any other resources that are needed by pods that run the Django application
         */
        const irsa = new irsa_1.Irsa(scope, 'Irsa', {
            cluster: this.cluster,
        });
        /**
         * Make sure that the namespace has been deployed
         */
        irsa.node.addDependency(appNamespace);
        irsa.chart.node.addDependency(appNamespace);
        /**
         * Lookup Certificate from ARN or generate
         * Deploy external-dns and related IAM resource if a domain name is included
         */
        if (props.domainName) {
            const hostedZone = route53.HostedZone.fromLookup(scope, 'hosted-zone', {
                domainName: props.domainName,
            });
            /**
             * Lookup or request ACM certificate depending on value of certificateArn
             */
            if (props.certificateArn) {
                // lookup ACM certificate from ACM certificate ARN
                acm.Certificate.fromCertificateArn(scope, 'certificate', props.certificateArn);
            }
            else {
                // request a new certificate
                new acm.Certificate(this, 'SSLCertificate', {
                    domainName: props.domainName,
                    validation: acm.CertificateValidation.fromDns(hostedZone),
                });
            }
            new external_dns_1.ExternalDns(scope, 'ExternalDns', {
                hostedZone,
                domainName: props.domainName,
                cluster: this.cluster,
            });
        }
        /**
         * RDS instance
         */
        const database = new database_1.RdsPostgresInstance(scope, 'RdsPostgresInstance', {
            vpc: this.vpc,
            // TODO: base dbSecret on environment name
            dbSecretName: 'dbSecret',
        });
        /**
         * Give the IRSA podRole read access to the bucket and read/write access to the S3 bucket
         */
        database.secret.grantRead(irsa.podRole);
        this.staticFileBucket.grantReadWrite(irsa.podRole);
        /**
         * Security Group for worker nodes
         */
        const appSecurityGroup = new ec2.SecurityGroup(scope, 'appSecurityGroup', {
            vpc: this.vpc,
        });
        /**
         * cluster.defaultCapactiy is the autoScalingGroup (the cluster's default node group)
         * Here the appSecurityGroup created above is added to that ASG
         *
         * TODO: use a non-default node-group
         */
        (_b = this.cluster.defaultCapacity) === null || _b === void 0 ? void 0 : _b.addSecurityGroup(appSecurityGroup);
        /**
         * Allow th ASG to accesss the database
         */
        database.rdsSecurityGroup.addIngressRule(appSecurityGroup, ec2.Port.tcp(5432));
        /**
         * Installation of AWS Load Balancer Controller
         * https://kubernetes-sigs.github.io/aws-load-balancer-controller/v2.2/deploy/installation/
         */
        const awslbc = new awslbc_1.AwsLoadBalancerController(this, 'AwsLoadBalancerController', {
            cluster: this.cluster,
        });
        /**
         * Backend docker image using DockerImageAssets
         * This image will be pushed to the ECR Registry created by `cdk bootstrap`
         */
        const backendImage = new ecrAssets.DockerImageAsset(scope, 'backendImage', {
            directory: props.imageDirectory,
        });
        /**
         * Common environment variables for Jobs and Deployments
         */
        const env = [
            {
                name: 'DEBUG',
                value: '0',
            },
            {
                // this is used in the application to fetch the secret value
                name: 'DB_SECRET_NAME',
                value: database.dbSecretName,
            },
            {
                name: 'POSTGRES_SERVICE_HOST',
                value: database.rdsPostgresInstance.dbInstanceEndpointAddress,
            },
            {
                name: 'DJANGO_SETTINGS_MODULE',
                value: 'backend.settings.production',
            },
        ];
        /**
         * Kubernetes Job that runs database migrations
         */
        const migrateJob = new migrate_1.MigrateJob(scope, 'django-migrate-job', {
            cluster: this.cluster,
            backendImage,
            namespace: 'app',
            env,
        });
        migrateJob.node.addDependency(appNamespace);
        // web service and deployment
        const webResources = new web_1.WebResources(scope, 'web-resources', {
            env,
            cluster: this.cluster,
            webCommand: (_c = props.webCommand) !== null && _c !== void 0 ? _c : ['./scripts/start_prod.sh'],
            backendImage,
            namespace: 'app',
        });
        webResources.node.addDependency(appNamespace);
        const webDeployment = this.cluster.addManifest('web-deployment', webResources.deploymentManifest);
        const webService = this.cluster.addManifest('web-service', webResources.serviceManifest);
        /**
         * This may not be necessary, but sometimes the namespace
         * is not found when deploying k8s resources
         */
        webDeployment.node.addDependency(appNamespace);
        webService.node.addDependency(appNamespace);
        /**
         * Add deployment and service manifests for web to the cluster
         */
        // const ingress = this.cluster.addManifest('app-ingresss', appIngress);
        const ingressResources = new ingress_1.AppIngressResources(scope, 'AppIngressResources', {
            cluster: this.cluster,
            domainName: 'test',
            certificateArn: props.certificateArn,
        });
        /**
         * Make sure that the Chart has been fully deployed before deploying the ingress,
         * otherwise there may be errors stating: no endpoints available for service "aws-load-balancer-webhook-service"
         */
        ingressResources.ingressManifest.node.addDependency(appNamespace);
        ingressResources.ingressManifest.node.addDependency(awslbc.chart);
        ingressResources.ingressManifest.node.addDependency(awslbc);
        ingressResources.ingressManifest.node.addDependency(webResources);
    }
}
exports.DjangoEks = DjangoEks;
_a = JSII_RTTI_SYMBOL_1;
DjangoEks[_a] = { fqn: "django-cdk.DjangoEks", version: "0.0.19" };
//# sourceMappingURL=data:application/json;base64,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