"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.managementCommandTask = void 0;
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * This construct allows you to define and optionally run a one-off command
 * each time the stack is created or updated. A Custom Resource is used to
 * invoke the AWS SDK to run a task if `run` is set to `true`.
 */
class managementCommandTask extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const taskDefinition = new ecs.TaskDefinition(scope, `TaskDefinitionFor${id}`, {
            compatibility: ecs.Compatibility.FARGATE,
            cpu: '256',
            memoryMiB: '512',
        });
        /**
         * Allow the task role to access the database
         */
        props.dbSecret.grantRead(taskDefinition.taskRole);
        /**
         * Add container to task
         */
        taskDefinition.addContainer(`TaskContainer${id}`, {
            image: props.image,
            command: props.command,
            environment: props.environment,
            logging: ecs.LogDriver.awsLogs({
                logRetention: logs.RetentionDays.ONE_DAY,
                streamPrefix: `${id}Container`,
            }),
        });
        // run the task
        if ((_a = props.run) !== null && _a !== void 0 ? _a : false) {
            // run the task in private subnets
            const subnets = props.vpc.privateSubnets.map(x => x.subnetId);
            // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/ECS.html
            const awsSdkCall = {
                action: 'runTask',
                service: 'ECS',
                parameters: {
                    cluster: props.cluster.clusterName,
                    taskDefinition: taskDefinition.taskDefinitionArn,
                    networkConfiguration: {
                        awsvpcConfiguration: {
                            subnets,
                            assignPublicIp: 'ENABLED',
                            securityGroups: [props.appSecurityGroup.securityGroupId],
                        },
                    },
                    count: 1,
                    launchType: 'FARGATE',
                },
                physicalResourceId: cr.PhysicalResourceId.of(id),
            };
            /**
             * Custom resource to run the ECS task using the awsSdkCall defined above
             *
             * Migrations are called when the stack is created or updated
             */
            new cr.AwsCustomResource(scope, 'CustomResourceDbMigration', {
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['iam:PassRole'],
                        resources: [
                            (_b = taskDefinition.executionRole) === null || _b === void 0 ? void 0 : _b.roleArn,
                            taskDefinition.taskRole.roleArn,
                        ],
                    }),
                    new iam.PolicyStatement({
                        actions: ['ecs:RunTask'],
                        resources: [taskDefinition.taskDefinitionArn],
                    }),
                ]),
                onCreate: awsSdkCall,
                onUpdate: awsSdkCall,
            });
        }
        ;
    }
}
exports.managementCommandTask = managementCommandTask;
//# sourceMappingURL=data:application/json;base64,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