# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['streamlit_webrtc']

package_data = \
{'': ['*'],
 'streamlit_webrtc': ['frontend/build/*', 'frontend/build/static/js/*']}

install_requires = \
['aiortc>=1.1.2,<2.0.0', 'streamlit>=0.84.1']

extras_require = \
{':python_version < "3.8"': ['typing_extensions>=3.7.4,<5.0.0']}

setup_kwargs = {
    'name': 'streamlit-webrtc',
    'version': '0.42.0',
    'description': '',
    'long_description': '# streamlit-webrtc\n**Handling and transmitting real-time video/audio streams over the network with Streamlit** [![Open in Streamlit](https://static.streamlit.io/badges/streamlit_badge_black_white.svg)](https://share.streamlit.io/whitphx/streamlit-webrtc-example/main/app.py)\n\n[![Tests](https://github.com/whitphx/streamlit-webrtc/workflows/Tests/badge.svg?branch=main)](https://github.com/whitphx/streamlit-webrtc/actions?query=workflow%3ATests+branch%3Amain)\n[![Frontend Tests](https://github.com/whitphx/streamlit-webrtc/workflows/Frontend%20tests/badge.svg?branch=main)](https://github.com/whitphx/streamlit-webrtc/actions?query=workflow%3A%22Frontend+tests%22+branch%3Amain)\n\n[![PyPI](https://img.shields.io/pypi/v/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n[![PyPI - License](https://img.shields.io/pypi/l/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n\n[![GitHub Sponsors](https://img.shields.io/github/sponsors/whitphx?label=Sponsor%20me%20on%20GitHub%20Sponsors&style=social)](https://github.com/sponsors/whitphx)\n\n<a href="https://www.buymeacoffee.com/whitphx" target="_blank"><img src="https://cdn.buymeacoffee.com/buttons/v2/default-yellow.png" alt="Buy Me A Coffee" width="180" height="50" ></a>\n\n<table>\n<tr>\n<td width="48%">\n  <a href="https://share.streamlit.io/whitphx/streamlit-webrtc-example/main/app.py">\n    <img src="https://aws1.discourse-cdn.com/business7/uploads/streamlit/original/2X/a/af111a7393c77cb69d7712ac8e71ca862feaeb24.gif" />\n  </a>\n</id>\n<td width="48%">\n  <a href="https://share.streamlit.io/whitphx/style-transfer-web-app/main/app.py">\n    <img src="https://aws1.discourse-cdn.com/business7/uploads/streamlit/original/2X/b/b3cb8aa60eb746366e06726a9137720583c02c3a.gif" />\n  </a>\n</id>\n</tr>\n</table>\n\n## Examples\n### [⚡️Showcase including following examples and more](https://github.com/whitphx/streamlit-webrtc-example): [🎈Online demo](https://share.streamlit.io/whitphx/streamlit-webrtc-example/main/app.py)\n\n* Object detection\n* OpenCV filter\n* Uni-directional video streaming\n* Audio processing\n\n### [⚡️Real-time Speech-to-Text](https://github.com/whitphx/streamlit-stt-app): [🎈Online demo](https://share.streamlit.io/whitphx/streamlit-stt-app/main/app_deepspeech.py)\n\nIt converts your voice into text in real time.\nThis app is self-contained; it does not depend on any external API.\n\n### [⚡️Real-time video style transfer](https://github.com/whitphx/style-transfer-web-app): [🎈Online demo](https://share.streamlit.io/whitphx/style-transfer-web-app/main/app.py)\nIt applies a wide variety of style transfer filters to real-time video streams.\n\n### [⚡️Video chat](https://github.com/whitphx/streamlit-video-chat-example)\n(Online demo not available)\n\nYou can create video chat apps with ~100 lines of Python code.\n\n### [⚡️Tokyo 2020 Pictogram](https://github.com/whitphx/Tokyo2020-Pictogram-using-MediaPipe): [🎈Online demo](https://share.streamlit.io/whitphx/tokyo2020-pictogram-using-mediapipe/streamlit-app)\n[MediaPipe](https://google.github.io/mediapipe/) is used for pose estimation.\n\n## Install\n```shell\n$ pip install -U streamlit-webrtc\n```\n\n## Quick tutorial\n\nSee also [the sample app, `app.py`](https://github.com/whitphx/streamlit-webrtc/blob/main/app.py), which contains a wide variety of usage.\n\nSee also ["Developing Web-Based Real-Time Video/Audio Processing Apps Quickly with Streamlit"](https://towardsdatascience.com/developing-web-based-real-time-video-audio-processing-apps-quickly-with-streamlit-7c7bcd0bc5a8).\n\n---\n\nCreate `app.py` with the content below.\n```py\nfrom streamlit_webrtc import webrtc_streamer\n\nwebrtc_streamer(key="sample")\n```\nUnlike other Streamlit components, `webrtc_streamer()` requires the `key` argument as a unique identifier. Set an arbitrary string to it.\n\nThen run it with Streamlit and open http://localhost:8501/.\n```shell\n$ streamlit run app.py\n```\n\nYou see the app view, so click the "START" button.\n\nThen, video and audio streaming starts. If asked for permissions to access the camera and microphone, allow it.\n![Basic example of streamlit-webrtc](./docs/images/streamlit_webrtc_basic.gif)\n\nNext, edit `app.py` as below and run it again.\n```py\nfrom streamlit_webrtc import webrtc_streamer\nimport av\n\n\ndef video_frame_callback(frame):\n    img = frame.to_ndarray(format="bgr24")\n\n    flipped = img[::-1,:,:]\n\n    return av.VideoFrame.from_ndarray(flipped, format="bgr24")\n\n\nwebrtc_streamer(key="example", video_frame_callback=video_frame_callback)\n```\n\nNow the video is vertically flipped.\n![Vertically flipping example](./docs/images/streamlit_webrtc_flipped.gif)\n\nAs an example above, you can edit the video frames by defining a callback that receives and returns a frame and passing it to the `video_frame_callback` argument (or `audio_frame_callback` for audio manipulation).\nThe input and output frames are the instance of [`av.VideoFrame`](https://pyav.org/docs/develop/api/video.html#av.video.frame.VideoFrame) (or [`av.AudioFrame`](https://pyav.org/docs/develop/api/audio.html#av.audio.frame.AudioFrame) when dealing with audio) of [`PyAV` library](https://pyav.org/).\n\nYou can inject any kinds of image (or audio) processing inside the callback.\nSee examples above for more applications.\n\n### Pass parameters to the callback\n\nYou can also pass parameters to the callback.\n\nIn the example below, a boolean `flip` flag is used to turn on/off the image flipping.\n```python\nimport streamlit as st\nfrom streamlit_webrtc import webrtc_streamer\nimport av\n\n\nflip = st.checkbox("Flip")\n\n\ndef video_frame_callback(frame):\n    img = frame.to_ndarray(format="bgr24")\n\n    flipped = img[::-1,:,:] if flip else img\n\n    return av.VideoFrame.from_ndarray(flipped, format="bgr24")\n\n\nwebrtc_streamer(key="example", video_frame_callback=video_frame_callback)\n```\n\n### Pull values from the callback\n\nSometimes we want to read the values generated in the callback from the outer scope.\n\nNote that the callback is executed in a forked thread running independently of the main script, so we have to take care of the following points and need some tricks for implementation like the example below (See also the section below for some limitations in the callback due to multi-threading).\n\n* Thread-safety\n  * Passing the values between inside and outside the callback must be thread-safe.\n* Using a loop to poll the values\n  * During media streaming, while the callback continues to be called, the main script execution stops at the bottom as usual. So we need to use a loop to keep the main script running and get the values from the callback in the outer scope.\n\nThe following example is to pass the image frames from the callback to the outer scope and continuously process it in the loop. In this example, a simple image analysis (calculating the histogram like [this OpenCV tutorial](https://docs.opencv.org/4.x/d1/db7/tutorial_py_histogram_begins.html)) is done on the image frames.\n\n[`threading.Lock`](https://docs.python.org/3/library/threading.html#lock-objects) is one standard way to control variable accesses across threads.\nA dict object `img_container` here is a mutable container shared by the callback and the outer scope and the `lock` object is used at assigning and reading the values to/from the container for thread-safety.\n\n```python\nimport threading\n\nimport cv2\nimport streamlit as st\nfrom matplotlib import pyplot as plt\n\nfrom streamlit_webrtc import webrtc_streamer\n\nlock = threading.Lock()\nimg_container = {"img": None}\n\n\ndef video_frame_callback(frame):\n    img = frame.to_ndarray(format="bgr24")\n    with lock:\n        img_container["img"] = img\n\n    return frame\n\n\nctx = webrtc_streamer(key="example", video_frame_callback=video_frame_callback)\n\nfig_place = st.empty()\nfig, ax = plt.subplots(1, 1)\n\nwhile ctx.state.playing:\n    with lock:\n        img = img_container["img"]\n    if img is None:\n        continue\n    gray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)\n    ax.cla()\n    ax.hist(gray.ravel(), 256, [0, 256])\n    fig_place.pyplot(fig)\n```\n\n## Callback limitations\nThe callbacks are executed in forked threads different from the main one, so there are some limitations:\n* Streamlit methods (`st.*` such as `st.write()`) do not work inside the callbacks.\n* Variables inside the callbacks cannot be directly referred to from the outside.\n* The `global` keyword does not work expectedly in the callbacks.\n* You have to care about thread-safety when accessing the same objects both from outside and inside the callbacks as stated in the section above.\n\n## Class-based callbacks\nUntil v0.37, the class-based callbacks were the standard.\nSee the [old version of the README](https://github.com/whitphx/streamlit-webrtc/blob/v0.37.0/README.md#quick-tutorial) about it.\n\n## Serving from remote host\nWhen deploying apps to remote servers, there are some things you need to be aware of.\n\n### HTTPS\n`streamlit-webrtc` uses [`getUserMedia()`](https://developer.mozilla.org/en-US/docs/Web/API/MediaDevices/getUserMedia) API to access local media devices, and this method does not work in an insecure context.\n\n[This document](https://developer.mozilla.org/en-US/docs/Web/API/MediaDevices/getUserMedia#privacy_and_security) says\n> A secure context is, in short, a page loaded using HTTPS or the file:/// URL scheme, or a page loaded from localhost.\n\nSo, when hosting your app on a remote server, it must be served via HTTPS if your app is using webcam or microphone.\nIf not, you will encounter an error when starting using the device. For example, it\'s something like below on Chrome.\n> Error: navigator.mediaDevices is undefined. It seems the current document is not loaded securely.\n\n[Streamlit Cloud](https://streamlit.io/cloud) is a recommended way for HTTPS serving. You can easily deploy Streamlit apps with it, and most importantly for this topic, it serves the apps via HTTPS automatically by defualt.\n\n### Configure the STUN server\nTo deploy the app to the cloud, we have to configure the *STUN* server via the `rtc_configuration` argument on `webrtc_streamer()` like below.\n\n```python\nwebrtc_streamer(\n    # ...\n    rtc_configuration={  # Add this config\n        "iceServers": [{"urls": ["stun:stun.l.google.com:19302"]}]\n    }\n    # ...\n)\n```\n\nThis configuration is necessary to establish the media streaming connection when the server is on a remote host.\n\n`streamlit_webrtc` uses WebRTC for its video and audio streaming. It has to access a "STUN server" in the global network for the remote peers (precisely, peers over the NATs) to establish WebRTC connections.\nAs we don\'t see the details about STUN servers here, please google it if interested with keywords such as STUN, TURN, or NAT traversal, or read these articles ([1](https://towardsdatascience.com/developing-web-based-real-time-video-audio-processing-apps-quickly-with-streamlit-7c7bcd0bc5a8#1cec), [2](https://dev.to/whitphx/python-webrtc-basics-with-aiortc-48id), [3](https://www.3cx.com/pbx/what-is-a-stun-server/)).\n\nThe example above is configured to use `stun.l.google.com:19302`, which is a free STUN server provided by Google.\n\nYou can also use any other STUN servers.\nFor example, [one user reported](https://github.com/whitphx/streamlit-webrtc/issues/283#issuecomment-889753789) that the Google\'s STUN server had a huge delay when using from China network, and the problem was solved by changing the STUN server.\n\nFor those who know about the browser WebRTC API: The value of the rtc_configuration argument will be passed to the [`RTCPeerConnection`](https://developer.mozilla.org/en-US/docs/Web/API/RTCPeerConnection/RTCPeerConnection) constructor on the frontend.\n\n### Configure the TURN server if necessary\nEven if the STUN server is properly configured, media streaming may not work in some network environments.\nFor example, in some office or public networks, there are firewalls which drop the WebRTC packets.\n\nIn such environments, setting up a [TURN server](https://webrtc.org/getting-started/turn-server) is a solution. See https://github.com/whitphx/streamlit-webrtc/issues/335#issuecomment-897326755.\n\n## Logging\nFor logging, this library uses the standard `logging` module and follows the practice described in [the official logging tutorial](https://docs.python.org/3/howto/logging.html#advanced-logging-tutorial). Then the logger names are the same as the module names - `streamlit_webrtc` or `streamlit_webrtc.*`.\n\nSo you can get the logger instance with `logging.getLogger("streamlit_webrtc")` through which you can control the logs from this library.\n\nFor example, if you want to set the log level on this library\'s logger as WARNING, you can use the following code.\n```python\nst_webrtc_logger = logging.getLogger("streamlit_webrtc")\nst_webrtc_logger.setLevel(logging.WARNING)\n```\n\nIn practice, `aiortc`, a third-party package this library is internally using, also emits many INFO level logs and you may want to control its logs too.\nYou can do it in the same way as below.\n```python\naioice_logger = logging.getLogger("aioice")\naioice_logger.setLevel(logging.WARNING)\n```\n\n## API changes\nCurrently there is no documentation about the interface. See the example [app.py](./app.py) for the usage.\nThe API is not finalized yet and can be changed without backward compatibility in the future releases until v1.0.\n\n### For users since versions `<0.20`\n`VideoTransformerBase` and its `transform` method have been marked as deprecated in v0.20.0. Please use `VideoProcessorBase#recv()` instead.\nNote that the signature of the `recv` method is different from the `transform` in that the `recv` has to return an instance of `av.VideoFrame` or `av.AudioFrame`.\n\nAlso, `webrtc_streamer()`\'s `video_transformer_factory` and `async_transform` arguments are deprecated, so use `video_processor_factory` and `async_processing` respectively.\n\nSee the samples in [app.py](./app.py) for their usage.\n\n## Resources\n* [Developing web-based real-time video/audio processing apps quickly with Streamlit](https://www.whitphx.info/posts/20211231-streamlit-webrtc-video-app-tutorial/)\n  * A tutorial for real-time video app development using `streamlit-webrtc`.\n  * Crosspost on dev.to: https://dev.to/whitphx/developing-web-based-real-time-videoaudio-processing-apps-quickly-with-streamlit-4k89\n* [New Component: streamlit-webrtc, a new way to deal with real-time media streams (Streamlit Community)](https://discuss.streamlit.io/t/new-component-streamlit-webrtc-a-new-way-to-deal-with-real-time-media-streams/8669)\n  * This is a forum topic where `streamlit-webrtc` has been introduced and discussed about.\n',
    'author': 'Yuichiro Tsuchiya',
    'author_email': 't.yic.yt@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/whitphx/streamlit-webrtc',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<3.11',
}


setup(**setup_kwargs)
