# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['visual_graph_datasets',
 'visual_graph_datasets.examples',
 'visual_graph_datasets.experiments',
 'visual_graph_datasets.generation',
 'visual_graph_datasets.processing',
 'visual_graph_datasets.visualization']

package_data = \
{'': ['*'], 'visual_graph_datasets': ['templates/*']}

install_requires = \
['cairosvg>=2.5.0',
 'click>=7.1.2',
 'imageio>=2.22.4',
 'jinja2>=3.0.3',
 'matplotlib>=3.5.3',
 'networkx>=2.8.8',
 'numpy>=1.23.2',
 'orjson>=3.8.0',
 'poetry-bumpversion>=0.3.0',
 'psutil>=5.7.2',
 'pycomex>=0.8.0',
 'pytest>=7.2.0',
 'pyyaml>=0.6.0',
 'rdkit>=2022.9.0',
 'scipy>=1.10.1']

setup_kwargs = {
    'name': 'visual-graph-datasets',
    'version': '0.10.1',
    'description': 'Datasets for the training of graph neural networks (GNNs) and subsequent visualization of attributional explanations of XAI methods',
    'long_description': '|made-with-python| |python-version| |os-linux|\n\n.. |os-linux| image:: https://img.shields.io/badge/os-linux-orange.svg\n   :target: https://www.python.org/\n\n.. |python-version| image:: https://img.shields.io/badge/Python-3.8.0-green.svg\n   :target: https://www.python.org/\n\n.. |made-with-kgcnn| image:: https://img.shields.io/badge/Made%20with-KGCNN-blue.svg\n   :target: https://github.com/aimat-lab/gcnn_keras\n\n.. |made-with-python| image:: https://img.shields.io/badge/Made%20with-Python-1f425f.svg\n   :target: https://www.python.org/\n\n.. image:: ./banner.png\n   :alt: banner image\n\n=====================\nVisual Graph Datasets\n=====================\n\nThis package supplies management and utilities for **graph datasets** used to train **graph neural networks**\nand more specifically aimed at **explainable AI (XAI)** methods\n\nW.r.t to the structure and management of these datasets this package employs a different philosophy. Instead of the\nusual minimal packaging to CSV files, a visual graph dataset (VGD) represents each **dataset as a folder** where\neach element is represented by two files:\n\n- A ``json`` file containing metadata information, including the **full graph representation**\n- A ``png`` file containing a canonical visualization of the graph.\n\nWe believe that providing a canonical graph representation as well as a canonical visualization will help to\nmake AI methods trained on such datasets more comparable. The canonical visualization and standard utilities for\nthe visualization of attributional XAI explanations specifically are aimed to improve the comparability and\nreproducability of XAI methods in the future.\n\nInstallation\n============\n\nFirst clone this repository:\n\n.. code-block:: console\n\n    git clone https://github/username/visual_graph_datasets.git\n\nThen install it like this:\n\n.. code-block:: console\n\n    cd visual_graph_datasets\n    pip3 install -e .\n\nCommand Line Interface\n======================\n\nDownload datasets\n-----------------\n\n    **NOTE**: We *strongly* encourage to store datasets on an SSD instead of an HDD, as this can make a\n    difference of multiple hours(!) when loading especially large datasets.\n\nDatasets can simply be downloaded by name by using the ``download`` command:\n\n.. code-block:: console\n\n    // Example for the dataset \'rb_dual_motifs\'\n    python3 -m visual_graph_datasets.cli download "rb_dual_motifs"\n\nBy default this dataset will be downloaded into the folder ``$HOME/.visual_graph_datasets/datasets``\nwhere HOME is the current users home directory.\n\nThe dataset download destination can be changed in a config file by using the ``config`` command:\n\n.. code-block:: console\n\n    python3 -m visual_graph_datasets.cli config\n\nThis command will open the config file at ``$HOME/.visual_graph_datasets/config.yaml`` using the systems\ndefault text editor.\n\nList available datasets\n-----------------------\n\nYou can display a list of all the currently available datasets of the current remote file share provider\nand some metadata information about them by using the command ``list``:\n\n.. code-block:: console\n\n    python3 -m visual_graph_datasets.cli list\n\nQuickstart\n==========\n\nThe datasets are mainly intended to be used in combination with other packages, but this package provides\nsome basic utilities to load and explore the datasets themselves within python programs.\n\n.. code-block:: python\n\n    import os\n    import typing as t\n    import matplotlib.pyplot as plt\n\n    from visual_graph_datasets.config import Config\n    from visual_graph_datasets.web import ensure_dataset\n    from visual_graph_datasets.data import load_visual_graph_dataset\n    from visual_graph_datasets.visualization.base import draw_image\n    from visual_graph_datasets.visualization.importances import plot_node_importances_border\n    from visual_graph_datasets.visualization.importances import plot_edge_importances_border\n\n    # This object will load the settings from the main config file. This config file contains options\n    # such as changing the default datasets folder and defining custom alternative file share providers\n    config = Config()\n    config.load()\n\n    # First of all we need to make sure that the dataset exists locally, this function will download it from\n    # the default file share provider if it does not exist.\n    ensure_dataset(\'rb_dual_motifs\', config)\n\n    # Afterwards we can be sure that the datasets exists and can now load it from the default datasets path.\n    # The data will be loaded as a dictionary whose int keys are the indices of the corresponding elements\n    # and the values are dictionaries which contain all the relevant data about the dataset element,\n    # (Dataset format is explained below)\n    dataset_path = os.path.join(config.get_datasets_path(), \'rb_dual_motifs\')\n    data_index_map: t.Dict[int, dict] = {}\n    _, data_index_map = load_visual_graph_dataset(dataset_path)\n\n    # Using this information we can visualize the ground truth importance explanation annotations for one\n    # element of the dataset like this.\n    index = 0\n    data = data_index_map[index]\n    # This is the dictionary which represents the graph structure of the dataset element. Descriptive\n    # string keys and numpy array values.\n    g = data[\'metadata\'][\'graph\']\n    fig, ax = plt.subplots(ncols=1, nrows=1, figsize=(10, 10))\n    draw_image(ax, image_path=data[\'image_path\'])\n    plot_node_importances_border(\n        ax=ax,\n        g=g,\n        node_positions=g[\'image_node_positions\'],\n        node_importances=g[\'node_importances_2\'][:, 0],\n    )\n    plot_edge_importances_border(\n        ax=ax,\n        g=g,\n        node_positions=g[\'image_node_positions\'],\n        edge_importances=g[\'edge_importances_2\'][:, 0],\n    )\n    fig_path = os.path.join(os.getcwd(), \'importances.pdf\')\n    fig.savefig(fig_path)\n\n\nDataset Format\n==============\n\nVisual Graph Datasets are represented as *folders* containing multiple files. The primary content of these\ndataset folders is made up of *2* files per element in the dataset:\n\n- **A PNG file**. This is the canonical visualization of the graph which can subsequently be used to create\n  explanation visualizations as well. The pixel position of each node in the graph is attached as metadata\n  of the graph representation.\n\n- **A JSON file**. Primarily contains the *full* graph representation consisting of node attributes, edge\n  attributes, an edge list etc. May also contain custom metadata for each graph depending on the dataset.\n\nAdditionally, a dataset folder may also contain a ``.meta.yml`` file which contains additional metadata about\nthe dataset as a whole.\n\nAlso, a dataset folder contains a python module ``process.py`` it contains the standalone implementation\nof the preprocessing procedure which turns a domain-specific graph representation (think of SMILES codes\nfor molecular graphs) into valid graph representations for that specific dataset.\nThis module can be imported and used directly from python code. Alternatively, the module can be used as a\nstandalone command line application for programming language agnostic preprocessing of elements.\n\nElement Metadata JSON\n---------------------\n\nOne such metadata file belonging to one element of the dataset may have the following nested structure:\n\n- ``target``: a 1d array containing the target values for the element. For classification this usually\n  a one-hot encoded vector of classes already. For multi-task regression this vector may have an\n  arbitrary number of continuous regression targets. For single-task regression this will still be a\n  vector, albeit with the shape (1, )\n- ``index``: The canonical index of this element within the dataset\n- (``train_split`` *optional*) A list of int indices, where each index represents a different split.\n  if the number "1" is for example part of this list, that means that the corresponding element is\n  considered to be part of the training set of split "1". What each particular split is may be described\n  in the documentation of the dataset.\n- (``test_split`` *optional*) A list of int indices, where each index represents a different split.\n  if the number "1" is for example part of this list, that means that the corresponding element is\n  considered to be part of the test set of the split "1".\n- ``graph``: A dictionary which contains the entire graph representation of this element.\n\n    - ``node_indices``: array of shape (V, 1) with the integer node indices.\n    - ``node_attributes``: array of shape (V, N)\n    - ``edge_indices``: array of shape (E, 2) which are the tuples of integer node indices that\n      determine edges\n    - ``edge_attributes``: array of shape (E, M)\n    - ``node_positions`` array of shape (V, 2) which are the xy positions of each node in pixel\n      values within the corresponding image visualization of the element. This is the crucial\n      information which is required to use the existing image representations to visualize attributional\n      explanations!\n    - (``node_importances_{K}_{suffix}`` *optional*) array of shape (V, K) containing ground truth node importance\n      explanations, which assign an importance value of 0 to 1 to each node of the graph across K channels.\n      One dataset element may have none or multiple such annotations with different suffixes\n      determining the number of explanation channels and origin.\n    - (``edge_importances_{K}_{suffix}`` *optional*) array of shape (E, K) containing ground truth edge importance\n      explanations, which assign an importance value of 0 to 1 to each edge of the graph across K channels.\n      One dataset element may have none or multiple such annotations with different suffixes\n      determining the number of explanation channels and origin.\n\nAssuming the following shape definitions:\n\n- V - the number of nodes in a graph\n- E - the number of edges in a graph\n- N - the number of node attributes / features associated with each node\n- M - the number of edge attributes / features associated with each edge\n- K - the number of importance channels\n\nDataset Metadata YML\n--------------------\n\nOne such metadata file may have the following nested structure. Additionally, it may also contain custom\nadditional fields depending on each dataset.\n\n- ``version``: A string determining the current version of the dataset\n- ``description``: Short string description of what the dataset is about (for example where the data came\n  from, what types of graphs it consists of, what the prediction target is etc.)\n- ``visualization_description``: String description of what can be seen in the visualization of the graph.\n  There are many different types of graphs out there which may have very domain specific visualizations.\n  This string should provide a short description of how the visualizations may be interpreted.\n- ``references``: A list of strings, where each string is a short description of online resources which are\n  relevant to the dataset, usually including a URL. This could for example include references to scientific\n  publications where a dataset was first introduced.\n- ``file_size``: The integer *accumulated* size of all the files that make up the dataset in bytes.\n- ``num_elements``: The integer number of elements in the dataset\n- ``num_targets``: The size of the prediction target vector\n- ``num_node_attributes``: The size of the node attribute vector\n- ``num_edge_attributes``: The size of the edge attribute vector\n\nDatasets\n========\n\nHere is a list of the datasets currently uploaded on the main file share provider.\n\nFor more information about the individual datasets use the ``list`` command in the CLI (see above).\n\n* TO BE DONE\n\n',
    'author': 'awa59kst120df',
    'author_email': 'awa59kst120df@gmail.com',
    'maintainer': 'awa59kst120df',
    'maintainer_email': 'awa59kst120df@gmail.com',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8',
}


setup(**setup_kwargs)
