"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const apiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: apiName,
            protocolType: 'HTTP',
            corsConfiguration,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
            });
        }
    }
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * Add a new stage.
     */
    addStage(id, options) {
        return new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
    }
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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