"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const integration_1 = require("./integration");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(key, path) {
        this.key = key;
        this.path = path;
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(`${method !== null && method !== void 0 ? method : 'ANY'} ${path}`, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey('$default');
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        let integration;
        const config = props.integration.bind(this);
        integration = new integration_1.HttpIntegration(this, `${this.node.id}-Integration`, {
            httpApi: props.httpApi,
            integrationType: config.type,
            integrationUri: config.uri,
            method: config.method,
            payloadFormatVersion: config.payloadFormatVersion,
        });
        const routeProps = {
            apiId: props.httpApi.httpApiId,
            routeKey: props.routeKey.key,
            target: `integrations/${integration.integrationId}`,
        };
        const route = new apigatewayv2_generated_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
}
exports.HttpRoute = HttpRoute;
//# sourceMappingURL=data:application/json;base64,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