"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
describe('HttpApi', () => {
    test('default', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api');
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            Name: 'api',
            ProtocolType: 'HTTP',
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Stage', {
            ApiId: stack.resolve(api.httpApiId),
            StageName: '$default',
            AutoDeploy: true,
        });
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Route');
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Integration');
        expect(api.url).toBeDefined();
    });
    test('unsetting createDefaultStage', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api', {
            createDefaultStage: false,
        });
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Stage');
        expect(api.url).toBeUndefined();
    });
    test('default integration', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api', {
            defaultIntegration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: '$default',
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Integration', {
            ApiId: stack.resolve(httpApi.httpApiId),
        });
    });
    test('addRoutes() configures the correct routes', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api');
        httpApi.addRoutes({
            path: '/pets',
            methods: [lib_1.HttpMethod.GET, lib_1.HttpMethod.PATCH],
            integration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'GET /pets',
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'PATCH /pets',
        });
    });
    test('addRoutes() creates the default method', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api');
        httpApi.addRoutes({
            path: '/pets',
            integration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'ANY /pets',
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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