"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../../lib");
describe('HttpProxyIntegration', () => {
    test('default', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpRoute(stack, 'HttpProxyRoute', {
            httpApi: api,
            integration: new lib_1.HttpProxyIntegration({
                url: 'some-target-url',
            }),
            routeKey: lib_1.HttpRouteKey.with('/pets'),
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationType: 'HTTP_PROXY',
            IntegrationUri: 'some-target-url',
            PayloadFormatVersion: '1.0',
            IntegrationMethod: 'ANY',
        });
    });
    test('method option is correctly recognized', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpRoute(stack, 'HttpProxyRoute', {
            httpApi: api,
            integration: new lib_1.HttpProxyIntegration({
                url: 'some-target-url',
                method: lib_1.HttpMethod.PATCH,
            }),
            routeKey: lib_1.HttpRouteKey.with('/pets'),
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationMethod: 'PATCH',
        });
    });
    test('CORS Configuration is correctly configured.', () => {
        const stack = new core_1.Stack();
        new lib_1.HttpApi(stack, 'HttpApi', {
            corsPreflight: {
                allowCredentials: true,
                allowHeaders: ['Authorization'],
                allowMethods: [lib_1.HttpMethod.GET, lib_1.HttpMethod.HEAD, lib_1.HttpMethod.OPTIONS, lib_1.HttpMethod.POST],
                allowOrigins: ['*'],
                maxAge: core_1.Duration.seconds(36400),
            },
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            CorsConfiguration: {
                AllowCredentials: true,
                AllowHeaders: ['Authorization'],
                AllowMethods: ['GET', 'HEAD', 'OPTIONS', 'POST'],
                AllowOrigins: ['*'],
                MaxAge: 36400,
            },
        });
    });
    test('CorsConfiguration is ABSENT when not specified.', () => {
        const stack = new core_1.Stack();
        new lib_1.HttpApi(stack, 'HttpApi');
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            CorsConfiguration: assert_1.ABSENT,
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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