"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./private/cross-region-support-stack");
const full_action_descriptor_1 = require("./private/full-action-descriptor");
const rich_action_1 = require("./private/rich-action");
const stage_1 = require("./private/stage");
const validation_1 = require("./private/validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 *
 * // create a pipeline
 * const pipeline = new Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ stageName: 'Source' });
 *
 * // add a source action to the stage
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   outputArtifactName: 'SourceArtifact',
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // @deprecated(v2): switch to default false
        this.crossAccountKeys = (_a = props.crossAccountKeys) !== null && _a !== void 0 ? _a : true;
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            let encryptionKey;
            if (this.crossAccountKeys) {
                encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                    // remove the key - there is a grace period of a few days before it's gone for good,
                    // that should be enough for any emergency access to the bucket artifacts
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                });
                // add an alias to make finding the key in the console easier
                new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                    aliasName: this.generateNameForDefaultBucketKeyAlias(),
                    targetKey: encryptionKey,
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                });
            }
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: encryptionKey ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.anyValue({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline.
     * @param id the logical ID of the returned pipeline construct.
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`).
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).parseArn(pipelineArn).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage.
     * @returns the newly created Stage
     */
    addStage(props) {
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Access one of the pipeline's stages by stage name.
     */
    stage(stageName) {
        for (const stage of this._stages) {
            if (stage.stageName === stageName) {
                return stage;
            }
        }
        throw new Error(`Pipeline does not contain a stage named '${stageName}'. Available stages: ${this._stages.map(s => s.stageName).join(', ')}`);
    }
    /**
     * (experimental) Returns all of the {@link CrossRegionSupportStack}s that were generated automatically when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @experimental
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        const richAction = new rich_action_1.RichAction(action, this);
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(richAction);
        // get the role for the given action, handling if it's cross-account
        const actionRole = this.getRoleForAction(stage, richAction, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(richAction.actionProperties.variablesNamespace);
        // bind the Action
        const actionConfig = richAction.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            // must be 'action', not 'richAction',
            // as those are returned by the IStage.actions property,
            // and it's important customers of Pipeline get the same instance
            // back as they added to the pipeline
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure.
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     *
     * @override true
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        if (!action.isCrossRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // The action has a specific region,
        // require the pipeline to have a known region as well.
        this.requireRegion();
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        const crossRegionSupport = this.obtainCrossRegionSupportFor(action);
        // the stack containing the replication bucket must be deployed before the pipeline
        core_1.Stack.of(this).addDependency(crossRegionSupport.stack);
        // The Pipeline role must be able to replicate to that bucket
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: action.effectiveRegion,
        };
    }
    /**
     * Get or create the cross-region support construct for the given action
     */
    obtainCrossRegionSupportFor(action) {
        // this method is never called for non cross-region actions
        const actionRegion = action.effectiveRegion;
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            const otherStack = action.resourceStack;
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        return crossRegionSupport;
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id, {
                    createKmsKey: this.crossAccountKeys,
                });
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.requireApp();
        const supportStackId = `cross-region-stack-${pipelineAccount}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
                createKmsKey: this.crossAccountKeys,
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = core_1.Names.uniqueId(this);
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        // (otherwise we can't configure cross-account trust policies)
        if (action.isCrossAccount) {
            const artifactBucket = this.ensureReplicationResourcesExistFor(action).artifactBucket;
            if (!artifactBucket.encryptionKey) {
                throw new Error(`Artifact Bucket must have a KMS Key to add cross-account action '${action.actionProperties.actionName}' ` +
                    `(pipeline account: '${renderEnvDimension(this.env.account)}', action account: '${renderEnvDimension(action.effectiveAccount)}'). ` +
                    'Create Pipeline with \'crossAccountKeys: true\' (or pass an existing Bucket with a key)');
            }
        }
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${core_1.Names.uniqueId(this)}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const pipelineStack = core_1.Stack.of(this);
        if (action.actionProperties.resource) {
            const resourceStack = core_1.Stack.of(action.actionProperties.resource);
            // check if resource is from a different account
            if (pipelineStack.account === resourceStack.account) {
                return undefined;
            }
            else {
                this._crossAccountSupport[resourceStack.account] = resourceStack;
                return resourceStack;
            }
        }
        if (!action.actionProperties.account) {
            return undefined;
        }
        const targetAccount = action.actionProperties.account;
        // check whether the account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
        }
        // check whether the pipeline account is a static string
        if (core_1.Token.isUnresolved(pipelineStack.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        if (pipelineStack.account === targetAccount) {
            return undefined;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.requireApp();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: action.actionProperties.region ? action.actionProperties.region : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
}
exports.Pipeline = Pipeline;
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
/**
 * Render an env dimension without showing the ugly stringified tokens
 */
function renderEnvDimension(s) {
    return core_1.Token.isUnresolved(s) ? '(current)' : s;
}
//# sourceMappingURL=data:application/json;base64,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