"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        const createKmsKey = (_a = props.createKmsKey) !== null && _a !== void 0 ? _a : true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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