"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientSettings = void 0;
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const languages_1 = require("../../languages");
// initialize logger
const logger = log4js_1.getLogger();
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ClientSettings extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.options = options;
        this.clientSettingsFilename =
            options.clientSettingsFilename ?? ".client-settings.json";
        this.clientSettingsPath = path.join(options.generatedCodeDir, this.clientSettingsFilename);
        // add client settings file to .gitignore
        this.project.addGitIgnore(this.clientSettingsPath);
        // if spec changed --> generate all
        // if forceGenerate is on --> generate all
        if (options.specChanged || options.forceGenerateCodeAndDocs) {
            logger.debug(`specChanged=${options.specChanged}, forceGenerate=${options.forceGenerateCodeAndDocs} :: all clients and docs will be generated`);
            this.clientLanguageConfigs = this.getClientLanguageConfigs(options.clientLanguages, options.defaultClientLanguage);
            this.documentationFormatConfigs = this.getDocumentationFormatConfigs(options.documentationFormats);
        }
        else {
            // spec didn't change and forceGenerate is off
            // load previously generated client config
            const clientSettingsPathAbs = path.join(project.outdir, this.clientSettingsPath);
            logger.trace(`Reading client settings from ${clientSettingsPathAbs}`);
            const existingClientConfig = util_1.tryReadFileSync(clientSettingsPathAbs);
            if (existingClientConfig != null) {
                const parsedClientConfig = JSON.parse(existingClientConfig);
                // previously generated client settings
                const prevClientLanguages = parsedClientConfig.clientLanguages;
                const prevDocFormats = parsedClientConfig.documentationFormats;
                const currClientLanguages = options.clientLanguages;
                const currDocFormats = options.documentationFormats;
                // keep only those lang/doc formats that were not present previously
                this.clientLanguageConfigs = this.getClientLanguageConfigs(currClientLanguages.filter((lang) => !prevClientLanguages.includes(lang)), options.defaultClientLanguage);
                this.documentationFormatConfigs = this.getDocumentationFormatConfigs(currDocFormats.filter((format) => !prevDocFormats.includes(format)));
            }
            else {
                logger.debug(`No generated client settings file found at ${clientSettingsPathAbs}. All languages and docs will be generated`);
                this.clientLanguageConfigs = this.getClientLanguageConfigs(options.clientLanguages, options.defaultClientLanguage);
                this.documentationFormatConfigs = this.getDocumentationFormatConfigs(options.documentationFormats);
            }
        }
    }
    getClientLanguageConfigs(clientLanguages, defaultLanguage) {
        const values = Object.values(languages_1.ClientLanguage);
        return values.map((currLang) => {
            return {
                clientLanguage: currLang,
                isDefault: defaultLanguage === currLang,
                generate: clientLanguages.includes(currLang),
            };
        });
    }
    getDocumentationFormatConfigs(documentationFormats) {
        const values = Object.values(languages_1.DocumentationFormat);
        return values.map((currFormat) => {
            return {
                documentationFormat: currFormat,
                generate: documentationFormats.includes(currFormat),
            };
        });
    }
    synthesize() {
        // store the client config
        // this helps to optimize to re-generate projects when new language/doc format is
        // introduced while the spec doesn't change
        logger.trace(`Generating settings file to ${this.clientSettingsPath}`);
        const clientSettingsJsonFile = new projen_1.JsonFile(this.project, this.clientSettingsPath, {
            obj: {
                clientLanguages: this.options.clientLanguages,
                documentationFormats: this.options.documentationFormats,
            },
            marker: true,
            editGitignore: false,
        });
        logger.debug(`Generated settings file to ${clientSettingsJsonFile.absolutePath}`);
    }
}
exports.ClientSettings = ClientSettings;
//# sourceMappingURL=data:application/json;base64,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