"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const logger = log4js_1.getLogger();
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    ...options.tsconfig?.compilerOptions,
                },
            },
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        logger.debug(`specDir = "${this.specDir}"`);
        logger.debug(`specFileName = "${this.specFileName}"`);
        logger.debug(`generatedCodeDir = "${this.generatedCodeDir}"`);
        logger.debug(`forceGenerateCodeAndDocs = ${this.forceGenerateCodeAndDocs}`);
        logger.debug(`apiSrcDir = "${this.apiSrcDir}"`);
        // Allow json files to be imported (for importing the parsed spec)
        this.tsconfig?.addInclude(`${this.srcdir}/**/*.json`);
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib", "cdk-nag");
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.srcdir, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Delete the lib directory prior to compilation to ensure latest parsed spec json file is copied
        this.preCompileTask.exec(`rm -rf ${this.libdir}`);
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.TYPESCRIPT);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.TYPESCRIPT,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.package.packageName,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: options.defaultReleaseBranch,
                packageManager: options.packageManager,
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedTypescriptClient = this.generatedClients[languages_1.ClientLanguage.TYPESCRIPT];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            apiSrcDir: path.join(this.srcdir, this.apiSrcDir),
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleFile(this, path.join(this.srcdir, "index.ts"), {
            contents: `export * from "./${this.apiSrcDir}";`,
        });
        new projen_1.SampleDir(this, path.join(this.srcdir, this.apiSrcDir), {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.7.9" };
//# sourceMappingURL=data:application/json;base64,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