import { Metric, MetricOptions } from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as logs from '@aws-cdk/aws-logs';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Elasticsearch version
 */
export declare class ElasticsearchVersion {
    readonly version: string;
    /** AWS Elasticsearch 1.5 */
    static readonly V1_5: ElasticsearchVersion;
    /** AWS Elasticsearch 2.3 */
    static readonly V2_3: ElasticsearchVersion;
    /** AWS Elasticsearch 5.1 */
    static readonly V5_1: ElasticsearchVersion;
    /** AWS Elasticsearch 5.3 */
    static readonly V5_3: ElasticsearchVersion;
    /** AWS Elasticsearch 5.5 */
    static readonly V5_5: ElasticsearchVersion;
    /** AWS Elasticsearch 5.6 */
    static readonly V5_6: ElasticsearchVersion;
    /** AWS Elasticsearch 6.0 */
    static readonly V6_0: ElasticsearchVersion;
    /** AWS Elasticsearch 6.2 */
    static readonly V6_2: ElasticsearchVersion;
    /** AWS Elasticsearch 6.3 */
    static readonly V6_3: ElasticsearchVersion;
    /** AWS Elasticsearch 6.4 */
    static readonly V6_4: ElasticsearchVersion;
    /** AWS Elasticsearch 6.5 */
    static readonly V6_5: ElasticsearchVersion;
    /** AWS Elasticsearch 6.7 */
    static readonly V6_7: ElasticsearchVersion;
    /** AWS Elasticsearch 6.8 */
    static readonly V6_8: ElasticsearchVersion;
    /** AWS Elasticsearch 7.1 */
    static readonly V7_1: ElasticsearchVersion;
    /** AWS Elasticsearch 7.4 */
    static readonly V7_4: ElasticsearchVersion;
    /** AWS Elasticsearch 7.7 */
    static readonly V7_7: ElasticsearchVersion;
    /**
     * Custom Elasticsearch version
     * @param version custom version number
     */
    static of(version: string): ElasticsearchVersion;
    /**
     *
     * @param version Elasticsearch version number
     */
    private constructor();
}
/**
 * Configures the capacity of the cluster such as the instance type and the
 * number of instances.
 */
export interface CapacityConfig {
    /**
     * The number of instances to use for the master node.
     *
     * @default - no dedicated master nodes
     */
    readonly masterNodes?: number;
    /**
     * The hardware configuration of the computer that hosts the dedicated master
     * node, such as `m3.medium.elasticsearch`. For valid values, see [Supported
     * Instance Types]
     * (https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default - r5.large.elasticsearch
     */
    readonly masterNodeInstanceType?: string;
    /**
     * The number of data nodes (instances) to use in the Amazon ES domain.
     *
     * @default - 1
     */
    readonly dataNodes?: number;
    /**
     * The instance type for your data nodes, such as
     * `m3.medium.elasticsearch`. For valid values, see [Supported Instance
     * Types](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default - r5.large.elasticsearch
     */
    readonly dataNodeInstanceType?: string;
}
/**
 * Specifies zone awareness configuration options.
 */
export interface ZoneAwarenessConfig {
    /**
     * Indicates whether to enable zone awareness for the Amazon ES domain.
     * When you enable zone awareness, Amazon ES allocates the nodes and replica
     * index shards that belong to a cluster across two Availability Zones (AZs)
     * in the same region to prevent data loss and minimize downtime in the event
     * of node or data center failure. Don't enable zone awareness if your cluster
     * has no replica index shards or is a single-node cluster. For more information,
     * see [Configuring a Multi-AZ Domain]
     * (https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-managedomains-multiaz)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default - false
     */
    readonly enabled?: boolean;
    /**
     * If you enabled multiple Availability Zones (AZs), the number of AZs that you
     * want the domain to use. Valid values are 2 and 3.
     *
     * @default - 2 if zone awareness is enabled.
     */
    readonly availabilityZoneCount?: number;
}
/**
 * The configurations of Amazon Elastic Block Store (Amazon EBS) volumes that
 * are attached to data nodes in the Amazon ES domain. For more information, see
 * [Configuring EBS-based Storage]
 * (https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs)
 * in the Amazon Elasticsearch Service Developer Guide.
 */
export interface EbsOptions {
    /**
     * Specifies whether Amazon EBS volumes are attached to data nodes in the
     * Amazon ES domain.
     *
     * @default - true
     */
    readonly enabled?: boolean;
    /**
     * The number of I/O operations per second (IOPS) that the volume
     * supports. This property applies only to the Provisioned IOPS (SSD) EBS
     * volume type.
     *
     * @default - iops are not set.
     */
    readonly iops?: number;
    /**
     * The size (in GiB) of the EBS volume for each data node. The minimum and
     * maximum size of an EBS volume depends on the EBS volume type and the
     * instance type to which it is attached.  For more information, see
     * [Configuring EBS-based Storage]
     * (https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default 10
     */
    readonly volumeSize?: number;
    /**
     * The EBS volume type to use with the Amazon ES domain, such as standard, gp2, io1, st1, or sc1.
     * For more information, see[Configuring EBS-based Storage]
     * (https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default gp2
     */
    readonly volumeType?: ec2.EbsDeviceVolumeType;
}
/**
 * Configures log settings for the domain.
 */
export interface LoggingOptions {
    /**
     * Specify if slow search logging should be set up.
     * Requires Elasticsearch version 5.1 or later.
     *
     * @default - false
     */
    readonly slowSearchLogEnabled?: boolean;
    /**
     * Log slow searches to this log group.
     *
     * @default - a new log group is created if slow search logging is enabled
     */
    readonly slowSearchLogGroup?: logs.ILogGroup;
    /**
     * Specify if slow index logging should be set up.
     * Requires Elasticsearch version 5.1 or later.
     *
     * @default - false
     */
    readonly slowIndexLogEnabled?: boolean;
    /**
     * Log slow indices to this log group.
     *
     * @default - a new log group is created if slow index logging is enabled
     */
    readonly slowIndexLogGroup?: logs.ILogGroup;
    /**
     * Specify if Elasticsearch application logging should be set up.
     * Requires Elasticsearch version 5.1 or later.
     *
     * @default - false
     */
    readonly appLogEnabled?: boolean;
    /**
     * Log Elasticsearch application logs to this log group.
     *
     * @default - a new log group is created if app logging is enabled
     */
    readonly appLogGroup?: logs.ILogGroup;
}
/**
 * Whether the domain should encrypt data at rest, and if so, the AWS Key
 * Management Service (KMS) key to use. Can only be used to create a new domain,
 * not update an existing one. Requires Elasticsearch version 5.1 or later.
 */
export interface EncryptionAtRestOptions {
    /**
     * Specify true to enable encryption at rest.
     *
     * @default - encryption at rest is disabled.
     */
    readonly enabled?: boolean;
    /**
     * Supply if using KMS key for encryption at rest.
     *
     * @default - uses default aws/es KMS key.
     */
    readonly kmsKey?: kms.IKey;
}
/**
 * Configures Amazon ES to use Amazon Cognito authentication for Kibana.
 * @see https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html
 */
export interface CognitoOptions {
    /**
     * The Amazon Cognito identity pool ID that you want Amazon ES to use for Kibana authentication.
     */
    readonly identityPoolId: string;
    /**
     * A role that allows Amazon ES to configure your user pool and identity pool. It must have the `AmazonESCognitoAccess` policy attached to it.
     *
     * @see https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html#es-cognito-auth-prereq
     */
    readonly role: iam.IRole;
    /**
     * The Amazon Cognito user pool ID that you want Amazon ES to use for Kibana authentication.
     */
    readonly userPoolId: string;
}
/**
 * The virtual private cloud (VPC) configuration for the Amazon ES domain. For
 * more information, see [VPC Support for Amazon Elasticsearch Service
 * Domains](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html)
 * in the Amazon Elasticsearch Service Developer Guide.
 */
export interface VpcOptions {
    /**
     * The list of security groups that are associated with the VPC endpoints
     * for the domain. If you don't provide a security group ID, Amazon ES uses
     * the default security group for the VPC. To learn more, see [Security Groups for your VPC]
     * (https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html) in the Amazon VPC
     * User Guide.
     */
    readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * Provide one subnet for each Availability Zone that your domain uses. For
     * example, you must specify three subnet IDs for a three Availability Zone
     * domain. To learn more, see [VPCs and Subnets]
     * (https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html) in the
     * Amazon VPC User Guide.
     */
    readonly subnets: ec2.ISubnet[];
}
/**
 * The minimum TLS version required for traffic to the domain.
 */
export declare enum TLSSecurityPolicy {
    /** Cipher suite TLS 1.0 */
    TLS_1_0 = "Policy-Min-TLS-1-0-2019-07",
    /** Cipher suite TLS 1.2 */
    TLS_1_2 = "Policy-Min-TLS-1-2-2019-07"
}
/**
 * Specifies options for fine-grained access control.
 */
export interface AdvancedSecurityOptions {
    /**
     * ARN for the master user. Only specify this or masterUserName, but not both.
     *
     * @default - fine-grained access control is disabled
     */
    readonly masterUserArn?: string;
    /**
     * Username for the master user. Only specify this or masterUserArn, but not both.
     *
     * @default - fine-grained access control is disabled
     */
    readonly masterUserName?: string;
    /**
     * Password for the master user.
     *
     * You can use `SecretValue.plainText` to specify a password in plain text or
     * use `secretsmanager.Secret.fromSecretAttributes` to reference a secret in
     * Secrets Manager.
     *
     * @default - A Secrets Manager generated password
     */
    readonly masterUserPassword?: cdk.SecretValue;
}
/**
 * Properties for an AWS Elasticsearch Domain.
 */
export interface DomainProps {
    /**
     * Domain Access policies.
     *
     * @default - No access policies.
     */
    readonly accessPolicies?: iam.PolicyStatement[];
    /**
     * Additional options to specify for the Amazon ES domain.
     *
     * @default - no advanced options are specified
     */
    readonly advancedOptions?: {
        [key: string]: (string);
    };
    /**
     * Configures Amazon ES to use Amazon Cognito authentication for Kibana.
     *
     * @default - Cognito not used for authentication to Kibana.
     */
    readonly cognitoKibanaAuth?: CognitoOptions;
    /**
     * Enforces a particular physical domain name.
     *
     * @default - A name will be auto-generated.
     */
    readonly domainName?: string;
    /**
     * The configurations of Amazon Elastic Block Store (Amazon EBS) volumes that
     * are attached to data nodes in the Amazon ES domain. For more information, see
     * [Configuring EBS-based Storage]
     * (https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default - 10 GiB General Purpose (SSD) volumes per node.
     */
    readonly ebs?: EbsOptions;
    /**
     * The cluster capacity configuration for the Amazon ES domain.
     *
     * @default - 1 r5.large.elasticsearch data node; no dedicated master nodes.
     */
    readonly capacity?: CapacityConfig;
    /**
     * The cluster zone awareness configuration for the Amazon ES domain.
     *
     * @default - no zone awareness (1 AZ)
     */
    readonly zoneAwareness?: ZoneAwarenessConfig;
    /**
     * The Elasticsearch version that your domain will leverage.
     */
    readonly version: ElasticsearchVersion;
    /**
     * Encryption at rest options for the cluster.
     *
     * @default - No encryption at rest
     */
    readonly encryptionAtRest?: EncryptionAtRestOptions;
    /**
     * Configuration log publishing configuration options.
     *
     * @default - No logs are published
     */
    readonly logging?: LoggingOptions;
    /**
     * Specify true to enable node to node encryption.
     * Requires Elasticsearch version 6.0 or later.
     *
     * @default - Node to node encryption is not enabled.
     */
    readonly nodeToNodeEncryption?: boolean;
    /**
     * The hour in UTC during which the service takes an automated daily snapshot
     * of the indices in the Amazon ES domain. Only applies for Elasticsearch
     * versions below 5.3.
     *
     * @default - Hourly automated snapshots not used
     */
    readonly automatedSnapshotStartHour?: number;
    /**
     * The virtual private cloud (VPC) configuration for the Amazon ES domain. For
     * more information, see [VPC Support for Amazon Elasticsearch Service
     * Domains](https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html)
     * in the Amazon Elasticsearch Service Developer Guide.
     *
     * @default - VPC not used
     */
    readonly vpcOptions?: VpcOptions;
    /**
     * True to require that all traffic to the domain arrive over HTTPS.
     *
     * @default - false
     */
    readonly enforceHttps?: boolean;
    /**
     * The minimum TLS version required for traffic to the domain.
     *
     * @default - TLSSecurityPolicy.TLS_1_0
     */
    readonly tlsSecurityPolicy?: TLSSecurityPolicy;
    /**
     * Specifies options for fine-grained access control.
     * Requires Elasticsearch version 6.7 or later. Enabling fine-grained access control
     * also requires encryption of data at rest and node-to-node encryption, along with
     * enforced HTTPS.
     *
     * @default - fine-grained access control is disabled
     */
    readonly fineGrainedAccessControl?: AdvancedSecurityOptions;
    /**
     * Configures the domain so that unsigned basic auth is enabled. If no master user is provided a default master user
     * with username `admin` and a dynamically generated password stored in KMS is created. The password can be retrieved
     * by getting `masterUserPassword` from the domain instance.
     *
     * Setting this to true will also add an access policy that allows unsigned
     * access, enable node to node encryption, encryption at rest. If conflicting
     * settings are encountered (like disabling encryption at rest) enabling this
     * setting will cause a failure.
     *
     * @default - false
     */
    readonly useUnsignedBasicAuth?: boolean;
}
/**
 * An interface that represents an Elasticsearch domain - either created with the CDK, or an existing one.
 */
export interface IDomain extends cdk.IResource {
    /**
     * Arn of the Elasticsearch domain.
     *
     * @attribute
     */
    readonly domainArn: string;
    /**
     * Domain name of the Elasticsearch domain.
     *
     * @attribute
     */
    readonly domainName: string;
    /**
     * Endpoint of the Elasticsearch domain.
     *
     * @attribute
     */
    readonly domainEndpoint: string;
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Domain.
     */
    metric(metricName: string, props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props?: MetricOptions): Metric;
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props?: MetricOptions): Metric;
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWriteBlocked(props?: MetricOptions): Metric;
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props?: MetricOptions): Metric;
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props?: MetricOptions): Metric;
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props?: MetricOptions): Metric;
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props?: MetricOptions): Metric;
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props?: MetricOptions): Metric;
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props?: MetricOptions): Metric;
}
/**
 * A new or imported domain.
 */
declare abstract class DomainBase extends cdk.Resource implements IDomain {
    abstract readonly domainArn: string;
    abstract readonly domainName: string;
    abstract readonly domainEndpoint: string;
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Domain.
     */
    metric(metricName: string, props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props?: MetricOptions): Metric;
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props?: MetricOptions): Metric;
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWriteBlocked(props?: MetricOptions): Metric;
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props?: MetricOptions): Metric;
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props?: MetricOptions): Metric;
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props?: MetricOptions): Metric;
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props?: MetricOptions): Metric;
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props?: MetricOptions): Metric;
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props?: MetricOptions): Metric;
    private grant;
}
/**
 * Reference to an Elasticsearch domain.
 */
export interface DomainAttributes {
    /**
     * The ARN of the Elasticsearch domain.
     */
    readonly domainArn: string;
    /**
     * The domain endpoint of the Elasticsearch domain.
     */
    readonly domainEndpoint: string;
}
/**
 * Provides an Elasticsearch domain.
 */
export declare class Domain extends DomainBase implements IDomain {
    /**
     * Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope: Construct, id: string, domainEndpoint: string): IDomain;
    /**
     * Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope: Construct, id: string, attrs: DomainAttributes): IDomain;
    readonly domainArn: string;
    readonly domainName: string;
    readonly domainEndpoint: string;
    /**
     * Log group that slow searches are logged to.
     *
     * @attribute
     */
    readonly slowSearchLogGroup?: logs.ILogGroup;
    /**
     * Log group that slow indices are logged to.
     *
     * @attribute
     */
    readonly slowIndexLogGroup?: logs.ILogGroup;
    /**
     * Log group that application logs are logged to.
     *
     * @attribute
     */
    readonly appLogGroup?: logs.ILogGroup;
    /**
     * Master user password if fine grained access control is configured.
     */
    readonly masterUserPassword?: cdk.SecretValue;
    private readonly domain;
    constructor(scope: Construct, id: string, props: DomainProps);
}
export {};
