"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * Elasticsearch version
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom Elasticsearch version
     * @param version custom version number
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/** AWS Elasticsearch 1.5 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/** AWS Elasticsearch 2.3 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/** AWS Elasticsearch 5.1 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/** AWS Elasticsearch 5.3 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/** AWS Elasticsearch 5.5 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/** AWS Elasticsearch 5.6 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/** AWS Elasticsearch 6.0 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/** AWS Elasticsearch 6.2 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/** AWS Elasticsearch 6.3 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/** AWS Elasticsearch 6.4 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/** AWS Elasticsearch 6.5 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/** AWS Elasticsearch 6.7 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/** AWS Elasticsearch 6.8 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/** AWS Elasticsearch 7.1 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/** AWS Elasticsearch 7.4 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/** AWS Elasticsearch 7.7 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * The minimum TLS version required for traffic to the domain.
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    /** Cipher suite TLS 1.2 */
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this Domain.
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatencyP99', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatencyP99', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Elasticsearch domain.
 */
class Domain extends DomainBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_r = props.vpcOptions) === null || _r === void 0 ? void 0 : _r.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master and data node instance types must end with ".elasticsearch".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_s = props.useUnsignedBasicAuth) !== null && _s !== void 0 ? _s : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_t = props.encryptionAtRest) === null || _t === void 0 ? void 0 : _t.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.stringValue({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_u = props.fineGrainedAccessControl) === null || _u === void 0 ? void 0 : _u.masterUserArn;
        const masterUserNameProps = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_y = (_x = props.encryptionAtRest) === null || _x === void 0 ? void 0 : _x.enabled) !== null && _y !== void 0 ? _y : (((_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_0 = props.nodeToNodeEncryption) !== null && _0 !== void 0 ? _0 : unsignedBasicAuthEnabled;
        const volumeSize = (_2 = (_1 = props.ebs) === null || _1 === void 0 ? void 0 : _1.volumeSize) !== null && _2 !== void 0 ? _2 : 10;
        const volumeType = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeType) !== null && _4 !== void 0 ? _4 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.enabled) !== null && _6 !== void 0 ? _6 : true;
        const enforceHttps = (_7 = props.enforceHttps) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_8 = props.logging) === null || _8 === void 0 ? void 0 : _8.slowIndexLogEnabled) || ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled)
                || ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_13 = props.logging.slowSearchLogGroup) !== null && _13 !== void 0 ? _13 : new logs.LogGroup(scope, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_15 = props.logging.slowIndexLogGroup) !== null && _15 !== void 0 ? _15 : new logs.LogGroup(scope, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_16 = props.logging) === null || _16 === void 0 ? void 0 : _16.appLogEnabled) {
            this.appLogGroup = (_17 = props.logging.appLogGroup) !== null && _17 !== void 0 ? _17 : new logs.LogGroup(scope, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, 'ESLogGroupPolicy', {
                policyName: 'ESLogPolicy',
                policyStatements: [logPolicyStatement],
            });
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_18 = props.ebs) === null || _18 === void 0 ? void 0 : _18.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_20 = (_19 = props.encryptionAtRest) === null || _19 === void 0 ? void 0 : _19.kmsKey) === null || _20 === void 0 ? void 0 : _20.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    enabled: this.appLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_21 = this.appLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
                },
                SEARCH_SLOW_LOGS: {
                    enabled: this.slowSearchLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_22 = this.slowSearchLogGroup) === null || _22 === void 0 ? void 0 : _22.logGroupArn,
                },
                INDEX_SLOW_LOGS: {
                    enabled: this.slowIndexLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_23 = this.slowIndexLogGroup) === null || _23 === void 0 ? void 0 : _23.logGroupArn,
                },
            },
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_24 = props.cognitoKibanaAuth) === null || _24 === void 0 ? void 0 : _24.identityPoolId,
                roleArn: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.role.roleArn,
                userPoolId: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_27 = props.tlsSecurityPolicy) !== null && _27 !== void 0 ? _27 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_28 = this.masterUserPassword) === null || _28 === void 0 ? void 0 : _28.toString(),
                    },
                }
                : undefined,
        });
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_29 = props.accessPolicies) !== null && _29 !== void 0 ? _29 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,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