"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
    jest.resetAllMocks();
});
const readActions = ['ESHttpGet', 'ESHttpHead'];
const writeActions = ['ESHttpDelete', 'ESHttpPost', 'ESHttpPut', 'ESHttpPatch'];
const readWriteActions = [
    ...readActions,
    ...writeActions,
];
test('minimal example renders correctly', () => {
    new lib_1.Domain(stack, 'Domain', { version: lib_1.ElasticsearchVersion.V7_1 });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        CognitoOptions: {
            Enabled: false,
        },
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10,
            VolumeType: 'gp2',
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterEnabled: false,
            InstanceCount: 1,
            InstanceType: 'r5.large.elasticsearch',
            ZoneAwarenessEnabled: false,
        },
        ElasticsearchVersion: '7.1',
        EncryptionAtRestOptions: {
            Enabled: false,
        },
        LogPublishingOptions: {
            ES_APPLICATION_LOGS: {
                Enabled: false,
            },
            SEARCH_SLOW_LOGS: {
                Enabled: false,
            },
            INDEX_SLOW_LOGS: {
                Enabled: false,
            },
        },
        NodeToNodeEncryptionOptions: {
            Enabled: false,
        },
    });
});
describe('log groups', () => {
    test('slowSearchLogEnabled should create a custom log group', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_4,
            logging: {
                slowSearchLogEnabled: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            LogPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    Enabled: false,
                },
                SEARCH_SLOW_LOGS: {
                    CloudWatchLogsLogGroupArn: {
                        'Fn::GetAtt': [
                            'SlowSearchLogsE00DC2E7',
                            'Arn',
                        ],
                    },
                    Enabled: true,
                },
                INDEX_SLOW_LOGS: {
                    Enabled: false,
                },
            },
        });
    });
    test('slowIndexLogEnabled should create a custom log group', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_4,
            logging: {
                slowIndexLogEnabled: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            LogPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    Enabled: false,
                },
                SEARCH_SLOW_LOGS: {
                    Enabled: false,
                },
                INDEX_SLOW_LOGS: {
                    CloudWatchLogsLogGroupArn: {
                        'Fn::GetAtt': [
                            'SlowIndexLogsAD49DED0',
                            'Arn',
                        ],
                    },
                    Enabled: true,
                },
            },
        });
    });
    test('appLogEnabled should create a custom log group', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_4,
            logging: {
                appLogEnabled: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            LogPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    CloudWatchLogsLogGroupArn: {
                        'Fn::GetAtt': [
                            'AppLogsC5DF83A6',
                            'Arn',
                        ],
                    },
                    Enabled: true,
                },
                SEARCH_SLOW_LOGS: {
                    Enabled: false,
                },
                INDEX_SLOW_LOGS: {
                    Enabled: false,
                },
            },
        });
    });
});
describe('grants', () => {
    test('"grantRead" allows read actions associated with this domain resource', () => {
        testGrant(readActions, (p, d) => d.grantRead(p));
    });
    test('"grantWrite" allows write actions associated with this domain resource', () => {
        testGrant(writeActions, (p, d) => d.grantWrite(p));
    });
    test('"grantReadWrite" allows read and write actions associated with this domain resource', () => {
        testGrant(readWriteActions, (p, d) => d.grantReadWrite(p));
    });
    test('"grantIndexRead" allows read actions associated with an index in this domain resource', () => {
        testGrant(readActions, (p, d) => d.grantIndexRead('my-index', p), false, ['/my-index', '/my-index/*']);
    });
    test('"grantIndexWrite" allows write actions associated with an index in this domain resource', () => {
        testGrant(writeActions, (p, d) => d.grantIndexWrite('my-index', p), false, ['/my-index', '/my-index/*']);
    });
    test('"grantIndexReadWrite" allows read and write actions associated with an index in this domain resource', () => {
        testGrant(readWriteActions, (p, d) => d.grantIndexReadWrite('my-index', p), false, ['/my-index', '/my-index/*']);
    });
    test('"grantPathRead" allows read actions associated with a given path in this domain resource', () => {
        testGrant(readActions, (p, d) => d.grantPathRead('my-index/my-path', p), false, ['/my-index/my-path']);
    });
    test('"grantPathWrite" allows write actions associated with a given path in this domain resource', () => {
        testGrant(writeActions, (p, d) => d.grantPathWrite('my-index/my-path', p), false, ['/my-index/my-path']);
    });
    test('"grantPathReadWrite" allows read and write actions associated with a given path in this domain resource', () => {
        testGrant(readWriteActions, (p, d) => d.grantPathReadWrite('my-index/my-path', p), false, ['/my-index/my-path']);
    });
    test('"grant" for an imported domain', () => {
        const domainEndpoint = 'https://test-domain-2w2x2u3tifly-jcjotrt6f7otem4sqcwbch3c4u.testregion.es.amazonaws.com';
        const domain = lib_1.Domain.fromDomainEndpoint(stack, 'Domain', domainEndpoint);
        const user = new iam.User(stack, 'user');
        domain.grantReadWrite(user);
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'es:ESHttpGet',
                            'es:ESHttpHead',
                            'es:ESHttpDelete',
                            'es:ESHttpPost',
                            'es:ESHttpPut',
                            'es:ESHttpPatch',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':es:testregion:1234:domain/test-domain-2w2x2u3tifly',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':es:testregion:1234:domain/test-domain-2w2x2u3tifly/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'userDefaultPolicy083DF682',
            Users: [
                {
                    Ref: 'user2C2B57AE',
                },
            ],
        });
    });
});
describe('metrics', () => {
    test('Can use metricClusterStatusRed on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricClusterStatusRed(), 'ClusterStatus.red', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricClusterStatusYellow on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricClusterStatusYellow(), 'ClusterStatus.yellow', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricFreeStorageSpace on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricFreeStorageSpace(), 'FreeStorageSpace', aws_cloudwatch_1.Statistic.MINIMUM);
    });
    test('Can use metricClusterIndexWriteBlocked on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricClusterIndexWriteBlocked(), 'ClusterIndexWriteBlocked', aws_cloudwatch_1.Statistic.MAXIMUM, core_1.Duration.minutes(1));
    });
    test('Can use metricNodes on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricNodes(), 'Nodes', aws_cloudwatch_1.Statistic.MINIMUM, core_1.Duration.hours(1));
    });
    test('Can use metricAutomatedSnapshotFailure on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricAutomatedSnapshotFailure(), 'AutomatedSnapshotFailure', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricCPUUtilization on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricCPUUtilization(), 'CPUUtilization', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricJVMMemoryPressure on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricJVMMemoryPressure(), 'JVMMemoryPressure', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricMasterCPUUtilization on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricMasterCPUUtilization(), 'MasterCPUUtilization', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricMasterJVMMemoryPressure on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricMasterJVMMemoryPressure(), 'MasterJVMMemoryPressure', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricKMSKeyError on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricKMSKeyError(), 'KMSKeyError', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricKMSKeyInaccessible on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricKMSKeyInaccessible(), 'KMSKeyInaccessible', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricSearchableDocuments on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricSearchableDocuments(), 'SearchableDocuments', aws_cloudwatch_1.Statistic.MAXIMUM);
    });
    test('Can use metricSearchLatency on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricSearchLatency(), 'SearchLatencyP99', 'p99');
    });
    test('Can use metricIndexingLatency on an Elasticsearch Domain', () => {
        testMetric((domain) => domain.metricIndexingLatency(), 'IndexingLatencyP99', 'p99');
    });
});
describe('import', () => {
    test('static fromDomainEndpoint(endpoint) allows importing an external/existing domain', () => {
        const domainName = 'test-domain-2w2x2u3tifly';
        const domainEndpoint = `https://${domainName}-jcjotrt6f7otem4sqcwbch3c4u.testregion.es.amazonaws.com`;
        const imported = lib_1.Domain.fromDomainEndpoint(stack, 'Domain', domainEndpoint);
        expect(imported.domainName).toEqual(domainName);
        expect(imported.domainArn).toMatch(RegExp(`es:testregion:1234:domain/${domainName}$`));
        expect(stack).not.toHaveResource('AWS::Elasticsearch::Domain');
    });
    test('static fromDomainAttributes(attributes) allows importing an external/existing domain', () => {
        const domainName = 'test-domain-2w2x2u3tifly';
        const domainArn = `es:testregion:1234:domain/${domainName}`;
        const domainEndpoint = `https://${domainName}-jcjotrt6f7otem4sqcwbch3c4u.testregion.es.amazonaws.com`;
        const imported = lib_1.Domain.fromDomainAttributes(stack, 'Domain', {
            domainArn,
            domainEndpoint,
        });
        expect(imported.domainName).toEqual(domainName);
        expect(imported.domainArn).toEqual(domainArn);
        expect(stack).not.toHaveResource('AWS::Elasticsearch::Domain');
    });
});
describe('advanced security options', () => {
    const masterUserArn = 'arn:aws:iam::123456789012:user/JohnDoe';
    const masterUserName = 'JohnDoe';
    const password = 'password';
    const masterUserPassword = core_1.SecretValue.plainText(password);
    test('enable fine-grained access control with a master user ARN', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_1,
            fineGrainedAccessControl: {
                masterUserArn,
            },
            encryptionAtRest: {
                enabled: true,
            },
            nodeToNodeEncryption: true,
            enforceHttps: true,
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            AdvancedSecurityOptions: {
                Enabled: true,
                InternalUserDatabaseEnabled: false,
                MasterUserOptions: {
                    MasterUserARN: masterUserArn,
                },
            },
            EncryptionAtRestOptions: {
                Enabled: true,
            },
            NodeToNodeEncryptionOptions: {
                Enabled: true,
            },
            DomainEndpointOptions: {
                EnforceHTTPS: true,
            },
        });
    });
    test('enable fine-grained access control with a master user name and password', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_1,
            fineGrainedAccessControl: {
                masterUserName,
                masterUserPassword,
            },
            encryptionAtRest: {
                enabled: true,
            },
            nodeToNodeEncryption: true,
            enforceHttps: true,
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            AdvancedSecurityOptions: {
                Enabled: true,
                InternalUserDatabaseEnabled: true,
                MasterUserOptions: {
                    MasterUserName: masterUserName,
                    MasterUserPassword: password,
                },
            },
            EncryptionAtRestOptions: {
                Enabled: true,
            },
            NodeToNodeEncryptionOptions: {
                Enabled: true,
            },
            DomainEndpointOptions: {
                EnforceHTTPS: true,
            },
        });
    });
    test('enable fine-grained access control with a master user name and dynamically generated password', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_1,
            fineGrainedAccessControl: {
                masterUserName,
            },
            encryptionAtRest: {
                enabled: true,
            },
            nodeToNodeEncryption: true,
            enforceHttps: true,
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            AdvancedSecurityOptions: {
                Enabled: true,
                InternalUserDatabaseEnabled: true,
                MasterUserOptions: {
                    MasterUserName: masterUserName,
                    MasterUserPassword: {
                        'Fn::Join': [
                            '',
                            [
                                '{{resolve:secretsmanager:',
                                {
                                    Ref: 'DomainMasterUserBFAFA7D9',
                                },
                                ':SecretString:password::}}',
                            ],
                        ],
                    },
                },
            },
            EncryptionAtRestOptions: {
                Enabled: true,
            },
            NodeToNodeEncryptionOptions: {
                Enabled: true,
            },
            DomainEndpointOptions: {
                EnforceHTTPS: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                GenerateStringKey: 'password',
            },
        });
    });
    test('enabling fine-grained access control throws with Elasticsearch < 6.7', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V6_5,
            fineGrainedAccessControl: {
                masterUserArn,
            },
            encryptionAtRest: {
                enabled: true,
            },
            nodeToNodeEncryption: true,
            enforceHttps: true,
        })).toThrow(/Fine-grained access control requires Elasticsearch version 6\.7 or later/);
    });
    test('enabling fine-grained access control throws without node-to-node encryption enabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            fineGrainedAccessControl: {
                masterUserArn,
            },
            encryptionAtRest: {
                enabled: true,
            },
            nodeToNodeEncryption: false,
            enforceHttps: true,
        })).toThrow(/Node-to-node encryption is required when fine-grained access control is enabled/);
    });
    test('enabling fine-grained access control throws without encryption-at-rest enabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            fineGrainedAccessControl: {
                masterUserArn,
            },
            encryptionAtRest: {
                enabled: false,
            },
            nodeToNodeEncryption: true,
            enforceHttps: true,
        })).toThrow(/Encryption-at-rest is required when fine-grained access control is enabled/);
    });
    test('enabling fine-grained access control throws without enforceHttps enabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            fineGrainedAccessControl: {
                masterUserArn,
            },
            encryptionAtRest: {
                enabled: true,
            },
            nodeToNodeEncryption: true,
            enforceHttps: false,
        })).toThrow(/Enforce HTTPS is required when fine-grained access control is enabled/);
    });
});
describe('custom error responses', () => {
    test('error when availabilityZoneCount does not match vpcOptions.subnets length', () => {
        const vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_4,
            zoneAwareness: {
                enabled: true,
                availabilityZoneCount: 2,
            },
            vpcOptions: {
                subnets: [
                    new aws_ec2_1.Subnet(stack, 'Subnet', {
                        availabilityZone: 'testaz',
                        cidrBlock: vpc.vpcCidrBlock,
                        vpcId: vpc.vpcId,
                    }),
                ],
                securityGroups: [],
            },
        })).toThrow(/you need to provide a subnet for each AZ you are using/);
    });
    test('error when master or data node instance types do not end with .elasticsearch', () => {
        const error = /instance types must end with ".elasticsearch"/;
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V7_4,
            capacity: {
                masterNodeInstanceType: 'c5.large',
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain2', {
            version: lib_1.ElasticsearchVersion.V7_4,
            capacity: {
                dataNodeInstanceType: 'c5.2xlarge',
            },
        })).toThrow(error);
    });
    test('error when elasticsearchVersion is unsupported/unknown', () => {
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.of('5.4'),
        })).toThrow(/Unknown Elasticsearch version: 5\.4/);
    });
    test('error when log publishing is enabled for elasticsearch version < 5.1', () => {
        const error = /logs publishing requires Elasticsearch version 5.1 or later/;
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V2_3,
            logging: {
                appLogEnabled: true,
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain2', {
            version: lib_1.ElasticsearchVersion.V1_5,
            logging: {
                slowSearchLogEnabled: true,
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain3', {
            version: lib_1.ElasticsearchVersion.V1_5,
            logging: {
                slowIndexLogEnabled: true,
            },
        })).toThrow(error);
    });
    test('error when encryption at rest is enabled for elasticsearch version < 5.1', () => {
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V2_3,
            encryptionAtRest: {
                enabled: true,
            },
        })).toThrow(/Encryption of data at rest requires Elasticsearch version 5.1 or later/);
    });
    test('error when cognito for kibana is enabled for elasticsearch version < 5.1', () => {
        const user = new iam.User(stack, 'user');
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V2_3,
            cognitoKibanaAuth: {
                identityPoolId: 'test-identity-pool-id',
                role: new iam.Role(stack, 'Role', { assumedBy: user }),
                userPoolId: 'test-user-pool-id',
            },
        })).toThrow(/Cognito authentication for Kibana requires Elasticsearch version 5.1 or later/);
    });
    test('error when C5, I3, M5, or R5 instance types are specified for elasticsearch version < 5.1', () => {
        const error = /C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later/;
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V2_3,
            capacity: {
                masterNodeInstanceType: 'c5.medium.elasticsearch',
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain2', {
            version: lib_1.ElasticsearchVersion.V1_5,
            capacity: {
                dataNodeInstanceType: 'i3.2xlarge.elasticsearch',
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain3', {
            version: lib_1.ElasticsearchVersion.V1_5,
            capacity: {
                dataNodeInstanceType: 'm5.2xlarge.elasticsearch',
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain4', {
            version: lib_1.ElasticsearchVersion.V1_5,
            capacity: {
                masterNodeInstanceType: 'r5.2xlarge.elasticsearch',
            },
        })).toThrow(error);
    });
    test('error when node to node encryption is enabled for elasticsearch version < 6.0', () => {
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V5_6,
            nodeToNodeEncryption: true,
        })).toThrow(/Node-to-node encryption requires Elasticsearch version 6.0 or later/);
    });
    test('error when i3 instance types are specified with EBS enabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V7_4,
            capacity: {
                dataNodeInstanceType: 'i3.2xlarge.elasticsearch',
            },
            ebs: {
                volumeSize: 100,
                volumeType: aws_ec2_1.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
            },
        })).toThrow(/I3 instance types do not support EBS storage volumes/);
    });
    test('error when m3, r3, or t2 instance types are specified with encryption at rest enabled', () => {
        const error = /M3, R3, and T2 instance types do not support encryption of data at rest/;
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V7_4,
            capacity: {
                masterNodeInstanceType: 'm3.2xlarge.elasticsearch',
            },
            encryptionAtRest: {
                enabled: true,
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain2', {
            version: lib_1.ElasticsearchVersion.V7_4,
            capacity: {
                dataNodeInstanceType: 'r3.2xlarge.elasticsearch',
            },
            encryptionAtRest: {
                enabled: true,
            },
        })).toThrow(error);
        expect(() => new lib_1.Domain(stack, 'Domain3', {
            version: lib_1.ElasticsearchVersion.V7_4,
            capacity: {
                masterNodeInstanceType: 't2.2xlarge.elasticsearch',
            },
            encryptionAtRest: {
                enabled: true,
            },
        })).toThrow(error);
    });
    test('error when t2.micro is specified with elasticsearch version > 2.3', () => {
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V6_7,
            capacity: {
                masterNodeInstanceType: 't2.micro.elasticsearch',
            },
        })).toThrow(/t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3/);
    });
    test('error when any instance type other than R3 and I3 are specified without EBS enabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V7_4,
            ebs: {
                enabled: false,
            },
            capacity: {
                masterNodeInstanceType: 'm5.large.elasticsearch',
            },
        })).toThrow(/EBS volumes are required when using instance types other than r3 or i3/);
    });
    test('error when availabilityZoneCount is not 2 or 3', () => {
        const vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        expect(() => new lib_1.Domain(stack, 'Domain1', {
            version: lib_1.ElasticsearchVersion.V7_4,
            vpcOptions: {
                subnets: [
                    new aws_ec2_1.Subnet(stack, 'Subnet1', {
                        availabilityZone: 'testaz1',
                        cidrBlock: vpc.vpcCidrBlock,
                        vpcId: vpc.vpcId,
                    }),
                    new aws_ec2_1.Subnet(stack, 'Subnet2', {
                        availabilityZone: 'testaz2',
                        cidrBlock: vpc.vpcCidrBlock,
                        vpcId: vpc.vpcId,
                    }),
                    new aws_ec2_1.Subnet(stack, 'Subnet3', {
                        availabilityZone: 'testaz3',
                        cidrBlock: vpc.vpcCidrBlock,
                        vpcId: vpc.vpcId,
                    }),
                    new aws_ec2_1.Subnet(stack, 'Subnet4', {
                        availabilityZone: 'testaz4',
                        cidrBlock: vpc.vpcCidrBlock,
                        vpcId: vpc.vpcId,
                    }),
                ],
                securityGroups: [],
            },
            zoneAwareness: {
                availabilityZoneCount: 4,
            },
        })).toThrow(/Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3/);
    });
});
test('can specify future version', () => {
    new lib_1.Domain(stack, 'Domain', { version: lib_1.ElasticsearchVersion.of('8.2') });
    expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
        ElasticsearchVersion: '8.2',
    });
});
describe('unsigned basic auth', () => {
    test('can create a domain with unsigned basic auth', () => {
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            useUnsignedBasicAuth: true,
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            AdvancedSecurityOptions: {
                Enabled: true,
                InternalUserDatabaseEnabled: true,
                MasterUserOptions: {
                    MasterUserName: 'admin',
                },
            },
            EncryptionAtRestOptions: {
                Enabled: true,
            },
            NodeToNodeEncryptionOptions: {
                Enabled: true,
            },
            DomainEndpointOptions: {
                EnforceHTTPS: true,
            },
        });
    });
    test('does not overwrite master user ARN configuration', () => {
        const masterUserArn = 'arn:aws:iam::123456789012:user/JohnDoe';
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            fineGrainedAccessControl: {
                masterUserArn,
            },
            useUnsignedBasicAuth: true,
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            AdvancedSecurityOptions: {
                Enabled: true,
                InternalUserDatabaseEnabled: false,
                MasterUserOptions: {
                    MasterUserARN: masterUserArn,
                },
            },
            EncryptionAtRestOptions: {
                Enabled: true,
            },
            NodeToNodeEncryptionOptions: {
                Enabled: true,
            },
            DomainEndpointOptions: {
                EnforceHTTPS: true,
            },
        });
    });
    test('does not overwrite master user name and password', () => {
        const masterUserName = 'JohnDoe';
        const password = 'password';
        const masterUserPassword = core_1.SecretValue.plainText(password);
        new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_1,
            fineGrainedAccessControl: {
                masterUserName,
                masterUserPassword,
            },
            useUnsignedBasicAuth: true,
        });
        expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
            AdvancedSecurityOptions: {
                Enabled: true,
                InternalUserDatabaseEnabled: true,
                MasterUserOptions: {
                    MasterUserName: masterUserName,
                    MasterUserPassword: password,
                },
            },
            EncryptionAtRestOptions: {
                Enabled: true,
            },
            NodeToNodeEncryptionOptions: {
                Enabled: true,
            },
            DomainEndpointOptions: {
                EnforceHTTPS: true,
            },
        });
    });
    test('fails to create a domain with unsigned basic auth when enforce HTTPS is disabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            useUnsignedBasicAuth: true,
            enforceHttps: false,
        })).toThrow(/You cannot disable HTTPS and use unsigned basic auth/);
    });
    test('fails to create a domain with unsigned basic auth when node to node encryption is disabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            useUnsignedBasicAuth: true,
            nodeToNodeEncryption: false,
        })).toThrow(/You cannot disable node to node encryption and use unsigned basic auth/);
    });
    test('fails to create a domain with unsigned basic auth when encryption at rest is disabled', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V7_7,
            useUnsignedBasicAuth: true,
            encryptionAtRest: { enabled: false },
        })).toThrow(/You cannot disable encryption at rest and use unsigned basic auth/);
    });
    test('using unsigned basic auth throws with Elasticsearch < 6.7', () => {
        expect(() => new lib_1.Domain(stack, 'Domain', {
            version: lib_1.ElasticsearchVersion.V6_5,
            useUnsignedBasicAuth: true,
        })).toThrow(/Using unsigned basic auth requires Elasticsearch version 6\.7 or later./);
    });
});
function testGrant(expectedActions, invocation, appliesToDomainRoot = true, paths = ['/*']) {
    const domain = new lib_1.Domain(stack, 'Domain', { version: lib_1.ElasticsearchVersion.V7_4 });
    const user = new iam.User(stack, 'user');
    invocation(user, domain);
    const action = expectedActions.length > 1 ? expectedActions.map(a => `es:${a}`) : `es:${expectedActions[0]}`;
    const domainArn = {
        'Fn::GetAtt': [
            'Domain66AC69E0',
            'Arn',
        ],
    };
    const resolvedPaths = paths.map(path => {
        return {
            'Fn::Join': [
                '',
                [
                    domainArn,
                    path,
                ],
            ],
        };
    });
    const resource = appliesToDomainRoot
        ? [domainArn, ...resolvedPaths]
        : resolvedPaths.length > 1
            ? resolvedPaths
            : resolvedPaths[0];
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: action,
                    Effect: 'Allow',
                    Resource: resource,
                },
            ],
            Version: '2012-10-17',
        },
        PolicyName: 'userDefaultPolicy083DF682',
        Users: [
            {
                Ref: 'user2C2B57AE',
            },
        ],
    });
}
function testMetric(invocation, metricName, statistic = aws_cloudwatch_1.Statistic.SUM, period = core_1.Duration.minutes(5)) {
    const domain = new lib_1.Domain(stack, 'Domain', { version: lib_1.ElasticsearchVersion.V7_4 });
    const metric = invocation(domain);
    expect(metric).toMatchObject({
        metricName,
        namespace: 'AWS/ES',
        period,
        statistic,
        dimensions: {
            ClientId: '1234',
        },
    });
    expect(metric.dimensions).toHaveProperty('DomainName');
}
//# sourceMappingURL=data:application/json;base64,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