# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_slugify_processor', 'django_slugify_processor.templatetags']

package_data = \
{'': ['*']}

install_requires = \
['Django>=2.2']

extras_require = \
{'test': ['django-extensions']}

setup_kwargs = {
    'name': 'django-slugify-processor',
    'version': '1.2.0a0',
    'description': 'pipeline for slugification edgecases in django',
    'long_description': '# django-slugify-processor\n\nCustom-[`slugify()`](https://docs.djangoproject.com/en/4.0/ref/utils/#django.utils.text.slugify)\nsupport for django.\n\n[![Python Package](https://img.shields.io/pypi/v/django-slugify-processor.svg)](https://pypi.org/project/django-slugify-processor/)\n[![Build Status](https://github.com/tony/django-slugify-processor/workflows/tests/badge.svg)](https://django-slugify-processor.git-pull.com/)\n[![Docs](https://github.com/tony/django-slugify-processor/workflows/docs/badge.svg)](https://github.com/tony/django-slugify-processor/actions?query=workflow%3Adocs)\n[![Code Coverage](https://codecov.io/gh/tony/django-slugify-processor/branch/master/graph/badge.svg)](https://codecov.io/gh/tony/django-slugify-processor)\n[![License](https://img.shields.io/github/license/tony/django-slugify-processor.svg)](https://github.com/tony/django-slugify-processor/blob/master/LICENSE)\n\n# What are slugs?\n\n_Slugs_ are URL\'s, typically generated from post titles, that you want to be both human readable and\na valid URL. They are SEO friendly.\n\nDjango provides a\n[slugify function](https://docs.djangoproject.com/en/4.0/ref/utils/#django.utils.text.slugify) in\n`django.utils.text.slugify` which is also made available as a\n[default filter](https://github.com/django/django/blob/4.0.0/django/template/defaultfilters.py#L247-255).\n\nDjango slugs can be automatically generated in django models via packages such as:\n\n- [django-autoslug](https://pypi.python.org/pypi/django-autoslug)\n  ([docs](https://pythonhosted.org/django-autoslug/))\n  ([github](https://github.com/neithere/django-autoslug))\n- [django-extensions](https://pypi.python.org/pypi/django-extensions) via\n  [AutoSlugField](https://django-extensions.readthedocs.io/en/latest/field_extensions.html)\n  ([docs](https://django-extensions.readthedocs.io/en/latest/))\n  ([github](https://github.com/django-extensions/django-extensions))\n\n# The problem\n\nThis project is based on an article from [devel.tech](https://devel.tech) covering\n[django\'s import strings](https://devel.tech/tips/n/djms3tTe/how-django-uses-deferred-imports-to-scale/).\n\nCorner cases exist with slugification. For instance: Corner cases exist with slugification. For\ninstance:\n\n| Term | [`django.utils.text.slugify`] | What you want |\n| ---- | ----------------------------- | ------------- |\n| C    | c (correct)                   | n/a           |\n| C++  | c                             | cpp           |\n| C#   | c                             | c-sharp       |\n\nTo make matters worse, if using a specialized model field like `AutoSlugField` from django-autoslug\nor django-extensions, the default behavior may be to name the slugs for C++ and C# to "c-1", "c-2"\nafter "c" is taken.\n\nHere\'s another case, acronyms / shorthands:\n\n| Term              | [`django.utils.text.slugify`] | What you (may) want |\n| ----------------- | ----------------------------- | ------------------- |\n| New York City     | new-york-city                 | nyc                 |\n| Y Combinator      | y-combinator                  | yc                  |\n| Portland          | portland                      | pdx                 |\n| Texas             | texas                         | tx                  |\n| $                 | \'\' (empty)                    | usd, aud, etc?      |\n| US$               | us                            | usd                 |\n| A$                | a                             | aud                 |\n| bitcoin           | bitcoin                       | btc                 |\n| United States     | united-states                 | usa                 |\n| League of Legends | league-of-legends             | league              |\n| Apple® iPod Touch | apple-ipod-touch              | ipod-touch          |\n\nEach website and niche has its own edge cases for slugs. So we need a solution that can scale, where\nyou can craft your own functions.\n\n# How django-slugify-processor helps\n\nThis builds on top of [`django.utils.text.slugify`] to handle your django project\'s edgecases. By\ndefault, django-slugify-processor will be a pass through to django\'s default behavior. Adding\nslugification functions via your Django project\'s settings file allows you to adjust.\n\n[`django.utils.text.slugify`]:\n  https://github.com/django/django/blob/4.0/django/template/defaultfilters.py#L232\n\n# Installation\n\n```console\n$ pip install django-slugify-processor\n```\n\n# Configure\n\nTo create a processor, create a function that accepts a string, and returns a string. Assume this is\n_project/app/slugify_processors.py_:\n\n```python\ndef my_processor(value):\n   value = value.replace(\'++\', \'pp\')\n   return value\n```\n\nInside of your settings, add a `SLUGIFY_PROCESSORS` list of strings that points to the function.\nAnything that\'s compatible with\n[import_string](https://docs.djangoproject.com/en/4.0/ref/utils/#django.utils.module_loading.import_string),\nin your settings file:\n\n```python\nSLUGIFY_PROCESSORS = [\n   \'project.app.slugify_processors.my_processor\'\n]\n```\n\n# Usage\n\n## In normal django code\n\nImport `slugify` from `django_slugify_processor.text`:\n\n```python\nfrom django_slugify_processor.text import slugify\n\nprint(slugify(\'C++\'))\n> \'cpp\'\n```\n\n## Template code\n\ndjango-slugify-processor is designed to override the built-in`slugify` filter.\n\n### via load\n\nYou can load by default via `{% load django_slugify_processor %}` in your template.\n\nIn your settings `INSTALLED_APPS`:\n\n```python\nINSTALLED_APPS = [\n    \'django_slugify_processor\'\n]\n```\n\nIn your template:\n\n```django\n{% load slugify_processor %}\n{{"C++"|slugify}}\n```\n\n### via built-in\n\nTo make this available in all templates, in the `OPTIONS` of your template engine, add\n`django_slugify_processor.template_tags`:\n\n```python\nTEMPLATES = [{\n    \'BACKEND\': \'django.template.backends.django.DjangoTemplates\',\n    \'OPTIONS\': {\n        \'builtins\': [\n            \'django_slugify_processor.templatetags.slugify_processor\',\n        ],\n    },\n}]\n```\n\nFrom within the template file:\n\n```django\n{{"C++"|slugify}}\n```\n\nOutput should be: cpp\n\n## Models\n\nFor the most up to date documentation, view the documetation for the plugin you\'re using (e.g.\ndjango-autoslug or django-extensions).\n\nTo use django-slugify-processor\'s `slugify` instead of django\'s default, there will be a field\noption to use the function.\n\n### django-extensions\n\nTested with 1.9.7 (2017-11-26):\n\n```python\nfrom django.db import models\n\nfrom django_extensions.db.fields import AutoSlugField\nfrom django_slugify_processors.text import slugify\n\nclass MyModel(models.Model):\n    title = models.CharField(max_length=255)\n    slug = AutoSlugField(\n        populate_from=\'title\',\n        slugify_function=slugify\n    )\n```\n\n### django-autoslug\n\nTested with 1.9.3 (2017-11-26):\n\n```python\nfrom django.db import models\n\nfrom autoslug import AutoSlugField\nfrom django_slugify_processors.text import slugify\n\nclass MyModel(models.Model):\n    title = models.CharField(max_length=255)\n    slug = AutoSlugField(\n        populate_from=\'title\',\n        slugify=slugify\n    )\n```\n\n# Credits\n\n- tox.ini based off DRF\'s (BSD 2-clause licensed)\n- yapf configuration based off RTD / devel.tech\'s (MIT-licensed)\n\n# Project details\n\n- python support >= 3.7, pypy3\n- django support 2.2, > 3.1,\n- Source https://github.com/tony/django-slugify-processor\n- Docs https://django-slugify-processor.git-pull.com\n- API https://django-slugify-processor.git-pull.com/api.html\n- Changelog https://django-slugify-processor.git-pull.com/history.html\n- Issues https://github.com/tony/django-slugify-processor/issues\n- Test Coverage https://codecov.io/gh/tony/django-slugify-processor\n- pypi https://pypi.python.org/pypi/django-slugify-processor\n- Open Hub https://www.openhub.net/p/django-slugify-processor\n- License MIT\n- git repo\n\n  ```bash\n  $ git clone https://github.com/tony/django-slugify-processor.git\n  ```\n\n## Development\n\nInstall stable:\n\n```console\n$ pip install django-slugify-processor\n```\n\nLocal installation:\n\n```console\n$ git clone https://github.com/tony/django-slugify-processor.git\n```\n\n```console\n$ cd ./django-slugify-processor\n```\n\n```console\n$ poetry shell\n```\n\n```console\n$ pipenv install\n```\n\nTest:\n\n```console\n$ make test\n```\n',
    'author': 'Tony Narlock',
    'author_email': 'tony@git-pull.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://django-slugify-processor.git-pull.com',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
