"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlbFargateServices = exports.LoadBalancerAccessibility = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const cdknag = require("./cdknag");
/**
 * The load balancer accessibility.
 *
 * @stability stable
 */
var LoadBalancerAccessibility;
(function (LoadBalancerAccessibility) {
    LoadBalancerAccessibility["EXTERNAL_ONLY"] = "EXTERNAL_ONLY";
    LoadBalancerAccessibility["INTERNAL_ONLY"] = "INTERNAL_ONLY";
})(LoadBalancerAccessibility = exports.LoadBalancerAccessibility || (exports.LoadBalancerAccessibility = {}));
/**
 * @stability stable
 */
class AlbFargateServices extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.hasInternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                this.hasInternalLoadBalancer = true;
            }
        });
        // create the access log bucket
        const accessLogBucket = new cdknag.AccessLogDeliveryBucket(this, 'AccessLogBucket').bucket;
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
            this.externalAlb.logAccessLogs(accessLogBucket, `${id}-extalblog`);
        }
        if (this.hasInternalLoadBalancer) {
            this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
                vpc: this.vpc,
                internetFacing: false,
            });
            this.internalAlb.logAccessLogs(accessLogBucket, `${id}-intalblog`);
        }
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
            containerInsights: true,
            executeCommandConfiguration: {
                logging: ecs.ExecuteCommandLogging.DEFAULT,
            },
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
            });
            this.service.push(svc);
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_f = (_e = t.scalingPolicy) === null || _e === void 0 ? void 0 : _e.maxCapacity) !== null && _f !== void 0 ? _f : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_h = (_g = t.scalingPolicy) === null || _g === void 0 ? void 0 : _g.targetCpuUtilization) !== null && _h !== void 0 ? _h : 50,
            });
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.listenerPort}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_k = (_j = t.scalingPolicy) === null || _j === void 0 ? void 0 : _j.requestPerTarget) !== null && _k !== void 0 ? _k : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                });
                // listener for the internal ALB
                new elbv2.ApplicationListener(this, `IntAlbListener${t.listenerPort}`, {
                    loadBalancer: this.internalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [inttg],
                });
                // extra scaling policy
                scaling.scaleOnRequestCount('RequestScaling2', {
                    requestsPerTarget: (_m = (_l = t.scalingPolicy) === null || _l === void 0 ? void 0 : _l.requestPerTarget) !== null && _m !== void 0 ? _m : 1000,
                    targetGroup: inttg,
                });
                inttg.addTarget(svc);
            }
        });
        // Route53
        const zoneName = (_d = (_c = props.route53Ops) === null || _c === void 0 ? void 0 : _c.zoneName) !== null && _d !== void 0 ? _d : 'svc.local';
        const externalAlbRecordName = (_f = (_e = props.route53Ops) === null || _e === void 0 ? void 0 : _e.externalAlbRecordName) !== null && _f !== void 0 ? _f : 'external';
        const internalAlbRecordName = (_h = (_g = props.route53Ops) === null || _g === void 0 ? void 0 : _g.internalAlbRecordName) !== null && _h !== void 0 ? _h : 'internal';
        const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
            zoneName,
            vpc: this.vpc,
        });
        if (this.hasInternalLoadBalancer) {
            new route53.ARecord(this, 'InternalAlbAlias', {
                zone,
                recordName: internalAlbRecordName,
                target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
            });
        }
        if (this.hasExternalLoadBalancer) {
            new route53.ARecord(this, 'ExternalAlbAlias', {
                zone,
                recordName: externalAlbRecordName,
                target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
            });
        }
        if (this.hasExternalLoadBalancer) {
            new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
        }
        if (this.hasInternalLoadBalancer) {
            new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
            new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
        }
        // add solution ID for the stack
        if (!cdk.Stack.of(this).templateOptions.description) {
            cdk.Stack.of(this).templateOptions.description = '(SO8030) - AWS CDK stack with serverless-container-constructs';
        }
        /**
         * suppress the cdk-nag rules
         */
        if (this.externalAlb) {
            let sg;
            sg = this.externalAlb.node.tryFindChild('SecurityGroup');
            cdknag.Suppress.securityGroup(sg, [
                {
                    id: 'AwsSolutions-EC23',
                    reason: 'public ALB requires 0.0.0.0/0 inbound access',
                },
            ]);
        }
        if (this.internalAlb) {
            let sg;
            sg = this.internalAlb.node.tryFindChild('SecurityGroup');
            cdknag.Suppress.securityGroup(sg, [
                {
                    id: 'AwsSolutions-EC23',
                    reason: 'internal ALB requires 0.0.0.0/0 inbound access',
                },
            ]);
        }
        props.tasks.forEach(t => {
            var _b, _c;
            let cfnPolicy = (_b = t.task.executionRole) === null || _b === void 0 ? void 0 : _b.node.tryFindChild('DefaultPolicy');
            cdknag.Suppress.iamPolicy(cfnPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'ecr:GetAuthorizationToken requires wildcard resource',
                },
            ]);
            cfnPolicy = (_c = t.task.taskRole) === null || _c === void 0 ? void 0 : _c.node.tryFindChild('DefaultPolicy');
            cdknag.Suppress.iamPolicy(cfnPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'task role with ECS exec support requires wildcard resource for ssmmessages. see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html',
                },
            ]);
        });
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.AlbFargateServices = AlbFargateServices;
_a = JSII_RTTI_SYMBOL_1;
AlbFargateServices[_a] = { fqn: "serverless-container-constructs.AlbFargateServices", version: "0.1.2" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', {
                maxAzs: 3,
                natGateways: 1,
                flowLogs: { flowLogs: {} },
            });
}
//# sourceMappingURL=data:application/json;base64,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