"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const index_1 = require("./index");
const main_1 = require("./main");
const app = new aws_cdk_lib_1.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new aws_cdk_lib_1.Stack(app, 'demo-stack', { env });
if (stack.node.tryGetContext('AWS_SOLUTIONS_CHECK')) {
    aws_cdk_lib_1.Aspects.of(app).add(new cdk_nag_1.AwsSolutionsChecks());
}
const orderTask = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
orderTask.addContainer('order', {
    image: aws_cdk_lib_1.aws_ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
    logging: new aws_cdk_lib_1.aws_ecs.AwsLogDriver({ streamPrefix: 'order-' }),
});
const customerTask = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: aws_cdk_lib_1.aws_ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
const productTask = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: aws_cdk_lib_1.aws_ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
const svc = new index_1.AlbFargateServices(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            listenerPort: 80,
            accessibility: main_1.LoadBalancerAccessibility.EXTERNAL_ONLY,
            task: orderTask,
            desiredCount: 2,
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            accessibility: main_1.LoadBalancerAccessibility.INTERNAL_ONLY,
            listenerPort: 8080,
            task: customerTask,
            desiredCount: 1,
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        { listenerPort: 9090, task: productTask, desiredCount: 1, accessibility: main_1.LoadBalancerAccessibility.INTERNAL_ONLY },
    ],
    route53Ops: {
        zoneName,
        externalElbRecordName: externalAlbRecordName,
        internalElbRecordName: internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, aws_cdk_lib_1.aws_ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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