import { aws_ec2 as ec2, aws_ecs as ecs } from 'aws-cdk-lib';
import { Construct } from 'constructs';
export interface BaseFargateServicesProps {
    readonly vpc?: ec2.IVpc;
    readonly tasks: FargateTaskProps[];
    readonly route53Ops?: Route53Options;
    /**
     * create a FARGATE_SPOT only cluster
     * @default false
     */
    readonly spot?: boolean;
    /**
     * Whether to enable ECS Exec support
     * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html
     * @default false
     */
    readonly enableExecuteCommand?: boolean;
    /**
     * The subnets to associate with the service.
     * @default -
     * {
     *       subnetType: ec2.SubnetType.PRIVATE,
     * }
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
}
/**
 * The load balancer accessibility.
 */
export declare enum LoadBalancerAccessibility {
    /**
     * register to external load balancer only
     */
    EXTERNAL_ONLY = "EXTERNAL_ONLY",
    /**
     * register to internal load balancer only
     */
    INTERNAL_ONLY = "INTERNAL_ONLY"
}
export interface FargateTaskProps {
    readonly task: ecs.FargateTaskDefinition;
    readonly listenerPort: number;
    /**
     * desired number of tasks for the service
     * @default 1
     */
    readonly desiredCount?: number;
    /**
     * service autoscaling policy
     * @default - { maxCapacity: 10, targetCpuUtilization: 50, requestsPerTarget: 1000 }
     */
    readonly scalingPolicy?: ServiceScalingPolicy;
    readonly capacityProviderStrategy?: ecs.CapacityProviderStrategy[];
    /**
     * Register the service to internal ELB, external ELB or both.
     * @default both
     */
    readonly accessibility?: LoadBalancerAccessibility;
}
export interface ServiceScalingPolicy {
    /**
     * max capacity for the service autoscaling
     * @default 10
     */
    readonly maxCapacity?: number;
    /**
     * target cpu utilization
     * @default 50
     */
    readonly targetCpuUtilization?: number;
    /**
     * request per target
     * @default 1000
     */
    readonly requestPerTarget?: number;
}
export interface Route53Options {
    /**
     * private zone name
     * @default svc.local
     */
    readonly zoneName?: string;
    /**
     * the external ELB record name
     * @default external
     */
    readonly externalElbRecordName?: string;
    /**
     * the internal ELB record name
     * @default internal
     */
    readonly internalElbRecordName?: string;
}
export declare abstract class BaseFargateServices extends Construct {
    readonly vpc: ec2.IVpc;
    /**
     * The service(s) created from the task(s)
     */
    readonly service: ecs.FargateService[];
    protected zoneName: string;
    protected hasExternalLoadBalancer: boolean;
    protected hasInternalLoadBalancer: boolean;
    protected vpcSubnets: ec2.SubnetSelection;
    /**
     * determine if vpcSubnets are all public ones
     */
    private isPublicSubnets;
    constructor(scope: Construct, id: string, props: BaseFargateServicesProps);
    private validateSubnets;
}
