"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const core_1 = require("@aws-cdk/core");
const validation = require("./private/validation");
/**
 * An output artifact of an action.
 *
 * Artifacts can be used as input by some actions.
 */
class Artifact {
    /**
     *
     */
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     *
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact.
     */
    static artifact(name) {
        return new Artifact(name);
    }
    /**
     *
     */
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     *
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     *
     * @param fileName The name of the file.
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact, such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     *
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents. Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     *
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     *
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    /**
     *
     */
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    /**
     *
     */
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    /**
     *
     */
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    /**
     *
     */
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.string({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.string({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.string({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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