"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNagApp", version: "0.12.11" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNag", version: "0.12.11" };
//# sourceMappingURL=data:application/json;base64,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