"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else if (key.endsWith(constants.UNINSTALLABLE_PACKAGE_SUFFIX)) {
            status.uninstallable = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const match = submoduleKeyRegexp(language).exec(key);
                if (match != null) {
                    const [, submodule, isUnsupported] = match;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language, 'md'))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.notSupportedKeySuffix(language, undefined, 'md'))) {
                    recordPerLanguage(language, "Unsupported" /* UNSUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.corruptAssemblyKeySuffix(language, undefined, 'md'))) {
                    recordPerLanguage(language, "CorruptAssembly" /* CORRUPT_ASSEMBLY */, name, majorVersion, fullName);
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    const reports = [];
    function createReport(reportKey, packageVersions) {
        const report = JSON.stringify(packageVersions, null, 2);
        const { buffer, contentEncoding } = compress_content_lambda_shared_1.compressContent(Buffer.from(report));
        console.log(`Uploading list to s3://${bucket}/${reportKey}`);
        reports.push(aws.s3().putObject({
            Body: buffer,
            Bucket: bucket,
            ContentEncoding: contentEncoding,
            ContentType: 'application/json',
            Expires: new Date(Date.now() + 300000),
            Key: reportKey,
            Metadata: {
                'Lambda-Run-Id': context.awsRequestId,
                'Lambda-Log-Group-Name': context.logGroupName,
                'Lambda-Log-Stream-Name': context.logStreamName,
            },
        }).promise());
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const uninstallable = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if (status.uninstallable) {
                uninstallable.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("UninstallablePackageCount" /* UNINSTALLABLE_PACKAGE_COUNT */, uninstallable.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
        createReport(constants.UNINSTALLABLE_PACKAGES_REPORT, uninstallable);
    })();
    for (const entry of Array.from(perLanguage.entries())) {
        await aws_embedded_metrics_1.metricScope((metrics) => async (language, data) => {
            console.log('');
            console.log('##################################################');
            console.log(`### Start of data for ${language}`);
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of [
                "Supported" /* SUPPORTED */,
                "Unsupported" /* UNSUPPORTED */,
                "Missing" /* MISSING */,
                "CorruptAssembly" /* CORRUPT_ASSEMBLY */,
            ]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    if ((forStatus === "Missing" /* MISSING */ && metricName === "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */)
                        || (forStatus === "CorruptAssembly" /* CORRUPT_ASSEMBLY */ && metricName === "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */)) {
                        // generate reports for missing/corrupt only for package versions granularity
                        const reportKey = forStatus === "Missing" /* MISSING */ ?
                            constants.missingDocumentationReport(language) :
                            constants.corruptAssemblyReport(language);
                        createReport(reportKey, filtered.map(([name]) => name).sort());
                    }
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.length} entries`);
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
            console.log(`### End of data for ${language}`);
            console.log('##################################################');
            console.log('');
        })(...entry);
    }
    for (const report of reports) {
        await report;
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder, 'md');
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
    ["CorruptAssembly" /* CORRUPT_ASSEMBLY */]: {
        ["packages" /* PACKAGES */]: "CorruptAssemblyPackageCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "CorruptAssemblyMajorVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "CorruptAssemblySubmoduleCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
            case "CorruptAssembly" /* CORRUPT_ASSEMBLY */:
                // If we already have a status, only override with if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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