"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const markdown_render_1 = require("jsii-docgen/lib/docgen/render/markdown-render");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                     ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const deleted = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const uploads = new Map();
        const deletions = new Map();
        let unprocessable = false;
        function markPackage(e, marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const upload = uploadFile(event.bucket, key, event.assembly.versionId, Buffer.from(e.message));
            uploads.set(key, upload);
        }
        async function unmarkPackage(marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const marked = await aws.s3ObjectExists(event.bucket, key);
            if (!marked) {
                return;
            }
            const deletion = deleteFile(event.bucket, key);
            deletions.set(key, deletion);
        }
        console.log(`Generating documentation for ${packageFqn}...`);
        try {
            const docs = await docgen.Documentation.forPackage(tarball, { name: assembly.name });
            // if the package used to not be installable, remove the marker for it.
            await unmarkPackage(constants.UNINSTALLABLE_PACKAGE_SUFFIX);
            for (const language of language_1.DocumentationLanguage.ALL) {
                if (event.languages && !event.languages[language.toString()]) {
                    console.log(`Skipping language ${language} as it was not requested!`);
                    continue;
                }
                const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                    metrics.setDimensions();
                    metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                    async function renderAndDispatch(submodule) {
                        try {
                            console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                            const json = await docs.toJson({
                                submodule,
                                language: docgen.Language.fromString(lang.name),
                            });
                            const jsonPage = Buffer.from(json.render(null, 2));
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, jsonPage.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: jsonBody, contentEncoding: jsonContentEncoding } = compress_content_lambda_shared_1.compressContent(jsonPage);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, jsonBody.length, aws_embedded_metrics_1.Unit.Bytes);
                            const jsonKey = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'json'));
                            console.log(`Uploading ${jsonKey}`);
                            const jsonUpload = uploadFile(event.bucket, jsonKey, event.assembly.versionId, jsonBody, jsonContentEncoding);
                            uploads.set(jsonKey, jsonUpload);
                            const markdown = markdown_render_1.MarkdownRenderer.fromSchema(json.content, {
                                anchorFormatter,
                                linkFormatter: linkFormatter(lang),
                            });
                            const page = Buffer.from(markdown.render());
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: body, contentEncoding } = compress_content_lambda_shared_1.compressContent(page);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                            const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'md'));
                            console.log(`Uploading ${key}`);
                            const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                            uploads.set(key, upload);
                            // if the package used to have a corrupt assembly, remove the marker for it.
                            await unmarkPackage(constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                        }
                        catch (e) {
                            if (e instanceof docgen.LanguageNotSupportedError) {
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'md'));
                            }
                            else if (e instanceof docgen.CorruptedAssemblyError) {
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                                unprocessable = true;
                            }
                            else {
                                throw e;
                            }
                        }
                    }
                    await renderAndDispatch();
                    for (const submodule of submodules) {
                        await renderAndDispatch(submodule);
                    }
                });
                await generateDocs(language);
            }
        }
        catch (error) {
            if (error instanceof docgen.UnInstallablePackageError) {
                markPackage(error, constants.UNINSTALLABLE_PACKAGE_SUFFIX);
                unprocessable = true;
            }
            else {
                throw error;
            }
        }
        for (const [key, upload] of uploads.entries()) {
            const response = await upload;
            created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
        }
        for (const [key, deletion] of deletions.entries()) {
            const response = await deletion;
            deleted.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished deleting ${key} (Version ID: ${response.VersionId})`);
        }
        if (unprocessable) {
            // the message here doesn't matter, we only use the error name
            // to divert this message away from the DLQ.
            const error = new Error();
            error.name = constants.UNPROCESSABLE_PACKAGE_ERROR_NAME;
        }
        // output must be compressed to satisfy 262,144 byte limit of SendTaskSuccess command
        const s3OKey = (s3Obj) => s3Obj.key;
        return { created: created.map(s3OKey), deleted: deleted.map(s3OKey) };
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: caching_1.CacheStrategy.default().toString(),
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
function deleteFile(bucket, key) {
    return aws.s3().deleteObject({
        Bucket: bucket,
        Key: key,
    }).promise();
}
function anchorFormatter(type) {
    const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
    const [base, ...rest] = name.split('.');
    if (rest.length > 0) {
        return sanitize(rest.join('.')); // Initializer.parameter.accessControl
    }
    else {
        return sanitize(base);
    }
}
function linkFormatter(lang) {
    const formatter = (type) => {
        const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
        const [baseName, ...rest] = name.split('.');
        const hash = '#' + rest.join('.'); // #Initializer.parameter.accessControl
        const langParam = `?lang=${lang.toString()}`;
        const submoduleParam = type.submodule ? `&submodule=${type.submodule}` : '';
        return `<a href="/packages/${type.packageName}/v/${type.packageVersion}/api/${baseName}${langParam}${submoduleParam}${hash}">${type.displayName}</a>`;
    };
    return formatter;
}
/**
 * Converts a type's id to an assembly-relative version, e.g.:
 * `aws-cdk-lib.aws_s3.Bucket.parameter.accessControl` => `Bucket.parameter.accessControl`
 */
function getAssemblyRelativeName(type) {
    let name = type.id;
    if (!name.startsWith(type.packageName)) {
        throw new Error(`Expected first part of "${type.id}" to start with "${type.packageName}".`);
    }
    name = name.slice(type.packageName.length + 1); // remove "aws-cdk-lib.""
    if (type.submodule) {
        if (!name.startsWith(type.submodule)) {
            throw new Error(`Expected second part of "${type.id}" to start with "${type.submodule}".`);
        }
        name = name.slice(type.submodule.length + 1); // remove "aws_s3."
    }
    return name;
}
;
function sanitize(str) {
    // HTML5 allows any characters in IDs except whitespace
    return str.replace(/ /g, '-');
}
//# sourceMappingURL=data:application/json;base64,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