"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3PrefixList = void 0;
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
/**
 * This obtains the PrefixLisId for the AWS-Managed prefix list of the S3
 * service in the stack's current region.
 */
class S3PrefixList extends core_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const describePrefixLists = {
            action: 'describePrefixLists',
            service: 'EC2',
            parameters: {
                Filters: [{
                        Name: 'prefix-list-name',
                        Values: [`com.amazonaws.${core_1.Stack.of(this).region}.s3`],
                    }],
            },
            outputPaths: ['PrefixLists.0.PrefixListId', 'PrefixLists.0.PrefixListName'],
            physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('PrefixLists.0.PrefixListId'),
        };
        const resource = new custom_resources_1.AwsCustomResource(this, 'Resource', {
            onCreate: describePrefixLists,
            onUpdate: describePrefixLists,
            // The ec2:DescribePrefixList action does not support resource scoping (understandably)
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE }),
            logRetention: aws_logs_1.RetentionDays.THREE_MONTHS,
            resourceType: 'Custom::S3-PrefixList',
        });
        this.prefixListId = resource.getResponseField('PrefixLists.0.PrefixListId');
        this.prefixListName = resource.getResponseField('PrefixLists.0.PrefixListName');
    }
}
exports.S3PrefixList = S3PrefixList;
//# sourceMappingURL=data:application/json;base64,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