"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const PACKAGE_LINKS = env_lambda_shared_1.requireEnv('PACKAGE_LINKS');
    const PACKAGE_TAGS = env_lambda_shared_1.requireEnv('PACKAGE_TAGS');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const { license, name, version } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        // Add custom links content to metdata for display on the frontend
        const allowedLinks = JSON.parse(PACKAGE_LINKS);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        // Add computed tags to metadata
        const packageTagsConfig = JSON.parse(PACKAGE_TAGS);
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        const metadata = {
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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