"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const tarball_lambda_shared_1 = require("../../backend/shared/tarball.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const DETAIL_TYPE = 'CodeArtifact Package Version State Change';
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    metrics.setNamespace(constants_lambda_shared_1.METRICS_NAMESPACE);
    metrics.setDimensions({
        [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: event.detail.domainOwner,
        [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: event.detail.domainName,
        [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: event.detail.repositoryName,
    });
    const packageName = event.detail.packageNamespace
        ? `@${event.detail.packageNamespace}/${event.detail.packageName}`
        : event.detail.packageName;
    if (event.detail.operationType === 'Deleted') {
        console.log(`Operation type is "Deleted" for ${packageName}@${event.detail.packageVersion}`);
        metrics.putMetric("DeletedCount" /* DELETED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const denied = denyList.lookup(packageName, event.detail.packageVersion);
    if (denied) {
        console.log(`Package ${packageName}@${event.detail.packageVersion} denied: ${JSON.stringify(denied, null, 2)}`);
        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const { asset, packageVersionRevision } = await aws.codeArtifact().getPackageVersionAsset({
        asset: 'package.tgz',
        domainOwner: event.detail.domainOwner,
        domain: event.detail.domainName,
        repository: event.detail.repositoryName,
        format: event.detail.packageFormat,
        namespace: event.detail.packageNamespace,
        package: event.detail.packageName,
        packageVersion: event.detail.packageVersion,
    }).promise();
    const tarball = Buffer.from(asset);
    const { assemblyJson, packageJson } = await tarball_lambda_shared_1.extractObjects(tarball, {
        assemblyJson: { path: 'package/.jsii' },
        packageJson: { path: 'package/package.json', required: true },
    });
    metrics.putMetric("NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */, assemblyJson ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (assemblyJson == null) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not contain a .jsii assembly`);
        return;
    }
    const metadata = JSON.parse(packageJson.toString('utf-8'));
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const eligibleLicense = licenseList.lookup((_a = metadata.license) !== null && _a !== void 0 ? _a : 'UNLICENSED');
    metrics.putMetric("IneligibleLicenseCount" /* INELIGIBLE_LICENSE */, eligibleLicense ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (!eligibleLicense) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not use allow-listed license: ${(_b = metadata.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
        return;
    }
    const stagingKey = `${packageName}/${event.detail.packageVersion}/${packageVersionRevision}/package.tgz`;
    await aws.s3().putObject({
        Bucket: stagingBucket,
        Key: stagingKey,
        Body: asset,
        ContentType: 'application/octet-stream',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
    const messageBase = {
        tarballUri: `s3://${stagingBucket}/${stagingKey}`,
        metadata: { resources: event.resources.join(', ') },
        time: event.time,
    };
    const message = {
        ...messageBase,
        integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
    };
    return aws.sqs().sendMessage({
        MessageAttributes: {
            AWS_REQUEST_ID: { DataType: 'String', StringValue: context.awsRequestId },
            LOG_GROUP_NAME: { DataType: 'String', StringValue: context.logGroupName },
            LOG_STREAM_NAME: { DataType: 'String', StringValue: context.logStreamName },
        },
        MessageBody: JSON.stringify(message),
        MessageDeduplicationId: event.detail.eventDeduplicationId,
        QueueUrl: queueUrl,
    }).promise();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZS1hcnRpZmFjdC1mb3J3YXJkZXIubGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3BhY2thZ2Utc291cmNlcy9jb2RlYXJ0aWZhY3QvY29kZS1hcnRpZmFjdC1mb3J3YXJkZXIubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtEQUF3RTtBQUN4RSxvRkFBNkU7QUFHN0UsdUZBQThFO0FBQzlFLDBGQUFvRjtBQUNwRiw4REFBOEQ7QUFDOUQsOEVBQW9FO0FBRXBFLDBGQUF5RTtBQUN6RSxzRkFBNEU7QUFDNUUsdUVBQW9KO0FBRXBKLE1BQU0sV0FBVyxHQUFHLDJDQUFvRCxDQUFDO0FBRXpFLG9DQUFhLENBQUMsbUJBQW1CLEdBQUcsc0JBQVksQ0FBQyxNQUFNLENBQUM7QUFFM0MsUUFBQSxPQUFPLEdBQUcsa0NBQVcsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQStELEVBQUUsT0FBZ0IsRUFBRSxFQUFFOztJQUMxSSxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUV4RCxNQUFNLGFBQWEsR0FBRyw4QkFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hELE1BQU0sUUFBUSxHQUFHLDhCQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7SUFFekMsT0FBTyxDQUFDLFlBQVksQ0FBQywyQ0FBaUIsQ0FBQyxDQUFDO0lBQ3hDLE9BQU8sQ0FBQyxhQUFhLENBQUM7UUFDcEIsQ0FBQyxnREFBc0IsQ0FBQyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsV0FBVztRQUNsRCxDQUFDLCtDQUFxQixDQUFDLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO1FBQ2hELENBQUMsbURBQXlCLENBQUMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLGNBQWM7S0FDekQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0I7UUFDL0MsQ0FBQyxDQUFDLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRTtRQUNqRSxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUM7SUFFN0IsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7UUFDNUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxtQ0FBbUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUM3RixPQUFPLENBQUMsU0FBUyxxQ0FBMkIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0QsT0FBTztLQUNSO0lBRUQsTUFBTSxRQUFRLEdBQUcsTUFBTSxxQ0FBYyxDQUFDLFNBQVMsRUFBRSxDQUFDO0lBQ2xELE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDekUsSUFBSSxNQUFNLEVBQUU7UUFDVixPQUFPLENBQUMsR0FBRyxDQUFDLFdBQVcsV0FBVyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsY0FBYyxZQUFZLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDaEgsT0FBTyxDQUFDLFNBQVMsNENBQStCLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQy9ELE9BQU87S0FDUjtJQUVELE1BQU0sRUFBRSxLQUFLLEVBQUUsc0JBQXNCLEVBQUUsR0FBRyxNQUFNLEdBQUcsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQztRQUN4RixLQUFLLEVBQUUsYUFBYTtRQUNwQixXQUFXLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXO1FBQ3JDLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVU7UUFDL0IsVUFBVSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsY0FBYztRQUN2QyxNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhO1FBQ2xDLFNBQVMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLGdCQUFnQjtRQUN4QyxPQUFPLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXO1FBQ2pDLGNBQWMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLGNBQWM7S0FDNUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBQ2IsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFNLENBQUMsQ0FBQztJQUVwQyxNQUFNLEVBQUUsWUFBWSxFQUFFLFdBQVcsRUFBRSxHQUFHLE1BQU0sc0NBQWMsQ0FBQyxPQUFPLEVBQUU7UUFDbEUsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRTtRQUN2QyxXQUFXLEVBQUUsRUFBRSxJQUFJLEVBQUUsc0JBQXNCLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRTtLQUM5RCxDQUFDLENBQUM7SUFDSCxPQUFPLENBQUMsU0FBUyxxREFBb0MsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3ZGLElBQUksWUFBWSxJQUFJLElBQUksRUFBRTtRQUN4QixPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksV0FBVyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsY0FBYyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ3pHLE9BQU87S0FDUjtJQUVELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBQzNELE1BQU0sV0FBVyxHQUFHLE1BQU0sd0NBQWlCLENBQUMsU0FBUyxFQUFFLENBQUM7SUFDeEQsTUFBTSxlQUFlLEdBQUcsV0FBVyxDQUFDLE1BQU0sT0FBQyxRQUFRLENBQUMsT0FBTyxtQ0FBSSxZQUFZLENBQUMsQ0FBQztJQUM3RSxPQUFPLENBQUMsU0FBUyxvREFBZ0MsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3RGLElBQUksQ0FBQyxlQUFlLEVBQUU7UUFDcEIsT0FBTyxDQUFDLEdBQUcsQ0FBQyxZQUFZLFdBQVcsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLGNBQWMsd0NBQXdDLE1BQUEsUUFBUSxDQUFDLE9BQU8sbUNBQUksWUFBWSxFQUFFLENBQUMsQ0FBQztRQUM5SSxPQUFPO0tBQ1I7SUFFRCxNQUFNLFVBQVUsR0FBRyxHQUFHLFdBQVcsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLGNBQWMsSUFBSSxzQkFBc0IsY0FBYyxDQUFDO0lBQ3pHLE1BQU0sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztRQUN2QixNQUFNLEVBQUUsYUFBYTtRQUNyQixHQUFHLEVBQUUsVUFBVTtRQUNmLElBQUksRUFBRSxLQUFLO1FBQ1gsV0FBVyxFQUFFLDBCQUEwQjtRQUN2QyxRQUFRLEVBQUU7WUFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtZQUN4QyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsYUFBYTtZQUMxQyxlQUFlLEVBQUUsT0FBTyxDQUFDLFlBQVk7U0FDdEM7S0FDRixDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7SUFFYixNQUFNLFdBQVcsR0FBRztRQUNsQixVQUFVLEVBQUUsUUFBUSxhQUFhLElBQUksVUFBVSxFQUFFO1FBQ2pELFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtRQUNuRCxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7S0FDakIsQ0FBQztJQUNGLE1BQU0sT0FBTyxHQUFtQjtRQUM5QixHQUFHLFdBQVc7UUFDZCxTQUFTLEVBQUUsbUNBQVMsQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDO0tBQzNDLENBQUM7SUFDRixPQUFPLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUM7UUFDM0IsaUJBQWlCLEVBQUU7WUFDakIsY0FBYyxFQUFFLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsT0FBTyxDQUFDLFlBQVksRUFBRTtZQUN6RSxjQUFjLEVBQUUsRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSxPQUFPLENBQUMsWUFBWSxFQUFFO1lBQ3pFLGVBQWUsRUFBRSxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLE9BQU8sQ0FBQyxhQUFhLEVBQUU7U0FDNUU7UUFDRCxXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUM7UUFDcEMsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxvQkFBb0I7UUFDekQsUUFBUSxFQUFFLFFBQVE7S0FDbkIsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0FBQ2YsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25maWd1cmF0aW9uLCBtZXRyaWNTY29wZSwgVW5pdCB9IGZyb20gJ2F3cy1lbWJlZGRlZC1tZXRyaWNzJztcbmltcG9ydCBFbnZpcm9ubWVudHMgZnJvbSAnYXdzLWVtYmVkZGVkLW1ldHJpY3MvbGliL2Vudmlyb25tZW50L0Vudmlyb25tZW50cyc7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQsIEV2ZW50QnJpZGdlRXZlbnQgfSBmcm9tICdhd3MtbGFtYmRhJztcblxuaW1wb3J0IHsgRGVueUxpc3RDbGllbnQgfSBmcm9tICcuLi8uLi9iYWNrZW5kL2RlbnktbGlzdC9jbGllbnQubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBMaWNlbnNlTGlzdENsaWVudCB9IGZyb20gJy4uLy4uL2JhY2tlbmQvbGljZW5zZS1saXN0L2NsaWVudC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9hd3MubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyByZXF1aXJlRW52IH0gZnJvbSAnLi4vLi4vYmFja2VuZC9zaGFyZWQvZW52LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgSW5nZXN0aW9uSW5wdXQgfSBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9pbmdlc3Rpb24taW5wdXQubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBpbnRlZ3JpdHkgfSBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9pbnRlZ3JpdHkubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBleHRyYWN0T2JqZWN0cyB9IGZyb20gJy4uLy4uL2JhY2tlbmQvc2hhcmVkL3RhcmJhbGwubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBNRVRSSUNTX05BTUVTUEFDRSwgTWV0cmljTmFtZSwgRE9NQUlOX09XTkVSX0RJTUVOU0lPTiwgRE9NQUlOX05BTUVfRElNRU5TSU9OLCBSRVBPU0lUT1JZX05BTUVfRElNRU5TSU9OIH0gZnJvbSAnLi9jb25zdGFudHMubGFtYmRhLXNoYXJlZCc7XG5cbmNvbnN0IERFVEFJTF9UWVBFID0gJ0NvZGVBcnRpZmFjdCBQYWNrYWdlIFZlcnNpb24gU3RhdGUgQ2hhbmdlJyBhcyBjb25zdDtcblxuQ29uZmlndXJhdGlvbi5lbnZpcm9ubWVudE92ZXJyaWRlID0gRW52aXJvbm1lbnRzLkxhbWJkYTtcblxuZXhwb3J0IGNvbnN0IGhhbmRsZXIgPSBtZXRyaWNTY29wZSgobWV0cmljcykgPT4gYXN5bmMgKGV2ZW50OiBFdmVudEJyaWRnZUV2ZW50PHR5cGVvZiBERVRBSUxfVFlQRSwgQ29kZUFydGlmYWN0RGV0YWlsPiwgY29udGV4dDogQ29udGV4dCkgPT4ge1xuICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQsIG51bGwsIDIpfWApO1xuXG4gIGNvbnN0IHN0YWdpbmdCdWNrZXQgPSByZXF1aXJlRW52KCdCVUNLRVRfTkFNRScpO1xuICBjb25zdCBxdWV1ZVVybCA9IHJlcXVpcmVFbnYoJ1FVRVVFX1VSTCcpO1xuXG4gIG1ldHJpY3Muc2V0TmFtZXNwYWNlKE1FVFJJQ1NfTkFNRVNQQUNFKTtcbiAgbWV0cmljcy5zZXREaW1lbnNpb25zKHtcbiAgICBbRE9NQUlOX09XTkVSX0RJTUVOU0lPTl06IGV2ZW50LmRldGFpbC5kb21haW5Pd25lcixcbiAgICBbRE9NQUlOX05BTUVfRElNRU5TSU9OXTogZXZlbnQuZGV0YWlsLmRvbWFpbk5hbWUsXG4gICAgW1JFUE9TSVRPUllfTkFNRV9ESU1FTlNJT05dOiBldmVudC5kZXRhaWwucmVwb3NpdG9yeU5hbWUsXG4gIH0pO1xuXG4gIGNvbnN0IHBhY2thZ2VOYW1lID0gZXZlbnQuZGV0YWlsLnBhY2thZ2VOYW1lc3BhY2VcbiAgICA/IGBAJHtldmVudC5kZXRhaWwucGFja2FnZU5hbWVzcGFjZX0vJHtldmVudC5kZXRhaWwucGFja2FnZU5hbWV9YFxuICAgIDogZXZlbnQuZGV0YWlsLnBhY2thZ2VOYW1lO1xuXG4gIGlmIChldmVudC5kZXRhaWwub3BlcmF0aW9uVHlwZSA9PT0gJ0RlbGV0ZWQnKSB7XG4gICAgY29uc29sZS5sb2coYE9wZXJhdGlvbiB0eXBlIGlzIFwiRGVsZXRlZFwiIGZvciAke3BhY2thZ2VOYW1lfUAke2V2ZW50LmRldGFpbC5wYWNrYWdlVmVyc2lvbn1gKTtcbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLkRFTEVURURfQ09VTlQsIDEsIFVuaXQuQ291bnQpO1xuICAgIHJldHVybjtcbiAgfVxuXG4gIGNvbnN0IGRlbnlMaXN0ID0gYXdhaXQgRGVueUxpc3RDbGllbnQubmV3Q2xpZW50KCk7XG4gIGNvbnN0IGRlbmllZCA9IGRlbnlMaXN0Lmxvb2t1cChwYWNrYWdlTmFtZSwgZXZlbnQuZGV0YWlsLnBhY2thZ2VWZXJzaW9uKTtcbiAgaWYgKGRlbmllZCkge1xuICAgIGNvbnNvbGUubG9nKGBQYWNrYWdlICR7cGFja2FnZU5hbWV9QCR7ZXZlbnQuZGV0YWlsLnBhY2thZ2VWZXJzaW9ufSBkZW5pZWQ6ICR7SlNPTi5zdHJpbmdpZnkoZGVuaWVkLCBudWxsLCAyKX1gKTtcbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLkRFTllfTElTVEVEX0NPVU5ULCAxLCBVbml0LkNvdW50KTtcbiAgICByZXR1cm47XG4gIH1cblxuICBjb25zdCB7IGFzc2V0LCBwYWNrYWdlVmVyc2lvblJldmlzaW9uIH0gPSBhd2FpdCBhd3MuY29kZUFydGlmYWN0KCkuZ2V0UGFja2FnZVZlcnNpb25Bc3NldCh7XG4gICAgYXNzZXQ6ICdwYWNrYWdlLnRneicsIC8vIEFsd2F5cyBuYW1lZCB0aGlzIHdheSBmb3IgbnBtIHBhY2thZ2VzIVxuICAgIGRvbWFpbk93bmVyOiBldmVudC5kZXRhaWwuZG9tYWluT3duZXIsXG4gICAgZG9tYWluOiBldmVudC5kZXRhaWwuZG9tYWluTmFtZSxcbiAgICByZXBvc2l0b3J5OiBldmVudC5kZXRhaWwucmVwb3NpdG9yeU5hbWUsXG4gICAgZm9ybWF0OiBldmVudC5kZXRhaWwucGFja2FnZUZvcm1hdCxcbiAgICBuYW1lc3BhY2U6IGV2ZW50LmRldGFpbC5wYWNrYWdlTmFtZXNwYWNlLFxuICAgIHBhY2thZ2U6IGV2ZW50LmRldGFpbC5wYWNrYWdlTmFtZSxcbiAgICBwYWNrYWdlVmVyc2lvbjogZXZlbnQuZGV0YWlsLnBhY2thZ2VWZXJzaW9uLFxuICB9KS5wcm9taXNlKCk7XG4gIGNvbnN0IHRhcmJhbGwgPSBCdWZmZXIuZnJvbShhc3NldCEpO1xuXG4gIGNvbnN0IHsgYXNzZW1ibHlKc29uLCBwYWNrYWdlSnNvbiB9ID0gYXdhaXQgZXh0cmFjdE9iamVjdHModGFyYmFsbCwge1xuICAgIGFzc2VtYmx5SnNvbjogeyBwYXRoOiAncGFja2FnZS8uanNpaScgfSxcbiAgICBwYWNrYWdlSnNvbjogeyBwYXRoOiAncGFja2FnZS9wYWNrYWdlLmpzb24nLCByZXF1aXJlZDogdHJ1ZSB9LFxuICB9KTtcbiAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5OT1RfSlNJSV9FTkFCTEVEX0NPVU5ULCBhc3NlbWJseUpzb24gPyAwIDogMSwgVW5pdC5Db3VudCk7XG4gIGlmIChhc3NlbWJseUpzb24gPT0gbnVsbCkge1xuICAgIGNvbnNvbGUubG9nKGBQYWNrYWdlIFwiJHtwYWNrYWdlTmFtZX1AJHtldmVudC5kZXRhaWwucGFja2FnZVZlcnNpb259XCIgZG9lcyBub3QgY29udGFpbiBhIC5qc2lpIGFzc2VtYmx5YCk7XG4gICAgcmV0dXJuO1xuICB9XG5cbiAgY29uc3QgbWV0YWRhdGEgPSBKU09OLnBhcnNlKHBhY2thZ2VKc29uLnRvU3RyaW5nKCd1dGYtOCcpKTtcbiAgY29uc3QgbGljZW5zZUxpc3QgPSBhd2FpdCBMaWNlbnNlTGlzdENsaWVudC5uZXdDbGllbnQoKTtcbiAgY29uc3QgZWxpZ2libGVMaWNlbnNlID0gbGljZW5zZUxpc3QubG9va3VwKG1ldGFkYXRhLmxpY2Vuc2UgPz8gJ1VOTElDRU5TRUQnKTtcbiAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTkVMSUdJQkxFX0xJQ0VOU0UsIGVsaWdpYmxlTGljZW5zZSA/IDAgOiAxLCBVbml0LkNvdW50KTtcbiAgaWYgKCFlbGlnaWJsZUxpY2Vuc2UpIHtcbiAgICBjb25zb2xlLmxvZyhgUGFja2FnZSBcIiR7cGFja2FnZU5hbWV9QCR7ZXZlbnQuZGV0YWlsLnBhY2thZ2VWZXJzaW9ufVwiIGRvZXMgbm90IHVzZSBhbGxvdy1saXN0ZWQgbGljZW5zZTogJHttZXRhZGF0YS5saWNlbnNlID8/ICdVTkxJQ0VOU0VEJ31gKTtcbiAgICByZXR1cm47XG4gIH1cblxuICBjb25zdCBzdGFnaW5nS2V5ID0gYCR7cGFja2FnZU5hbWV9LyR7ZXZlbnQuZGV0YWlsLnBhY2thZ2VWZXJzaW9ufS8ke3BhY2thZ2VWZXJzaW9uUmV2aXNpb259L3BhY2thZ2UudGd6YDtcbiAgYXdhaXQgYXdzLnMzKCkucHV0T2JqZWN0KHtcbiAgICBCdWNrZXQ6IHN0YWdpbmdCdWNrZXQsXG4gICAgS2V5OiBzdGFnaW5nS2V5LFxuICAgIEJvZHk6IGFzc2V0LFxuICAgIENvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vb2N0ZXQtc3RyZWFtJyxcbiAgICBNZXRhZGF0YToge1xuICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICdMYW1iZGEtUnVuLUlkJzogY29udGV4dC5hd3NSZXF1ZXN0SWQsXG4gICAgfSxcbiAgfSkucHJvbWlzZSgpO1xuXG4gIGNvbnN0IG1lc3NhZ2VCYXNlID0ge1xuICAgIHRhcmJhbGxVcmk6IGBzMzovLyR7c3RhZ2luZ0J1Y2tldH0vJHtzdGFnaW5nS2V5fWAsXG4gICAgbWV0YWRhdGE6IHsgcmVzb3VyY2VzOiBldmVudC5yZXNvdXJjZXMuam9pbignLCAnKSB9LFxuICAgIHRpbWU6IGV2ZW50LnRpbWUsXG4gIH07XG4gIGNvbnN0IG1lc3NhZ2U6IEluZ2VzdGlvbklucHV0ID0ge1xuICAgIC4uLm1lc3NhZ2VCYXNlLFxuICAgIGludGVncml0eTogaW50ZWdyaXR5KG1lc3NhZ2VCYXNlLCB0YXJiYWxsKSxcbiAgfTtcbiAgcmV0dXJuIGF3cy5zcXMoKS5zZW5kTWVzc2FnZSh7XG4gICAgTWVzc2FnZUF0dHJpYnV0ZXM6IHtcbiAgICAgIEFXU19SRVFVRVNUX0lEOiB7IERhdGFUeXBlOiAnU3RyaW5nJywgU3RyaW5nVmFsdWU6IGNvbnRleHQuYXdzUmVxdWVzdElkIH0sXG4gICAgICBMT0dfR1JPVVBfTkFNRTogeyBEYXRhVHlwZTogJ1N0cmluZycsIFN0cmluZ1ZhbHVlOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSB9LFxuICAgICAgTE9HX1NUUkVBTV9OQU1FOiB7IERhdGFUeXBlOiAnU3RyaW5nJywgU3RyaW5nVmFsdWU6IGNvbnRleHQubG9nU3RyZWFtTmFtZSB9LFxuICAgIH0sXG4gICAgTWVzc2FnZUJvZHk6IEpTT04uc3RyaW5naWZ5KG1lc3NhZ2UpLFxuICAgIE1lc3NhZ2VEZWR1cGxpY2F0aW9uSWQ6IGV2ZW50LmRldGFpbC5ldmVudERlZHVwbGljYXRpb25JZCxcbiAgICBRdWV1ZVVybDogcXVldWVVcmwsXG4gIH0pLnByb21pc2UoKTtcbn0pO1xuXG4vKipcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVhcnRpZmFjdC9sYXRlc3QvdWcvc2VydmljZS1ldmVudC1mb3JtYXQtZXhhbXBsZS5odG1sXG4gKi9cbmludGVyZmFjZSBDb2RlQXJ0aWZhY3REZXRhaWwge1xuICAvKipcbiAgICogVGhlIGRvbWFpbiB0aGF0IGNvbnRhaW5zIHRoZSByZXBvc2l0b3J5IHRoYXQgY29udGFpbnMgdGhlIHBhY2thZ2UuXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBV1MgYWNjb3VudCBJRCBvZiB0aGUgb3duZXIgb2YgdGhlIGRvbWFpbi5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk93bmVyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZXBvc2l0b3J5IHRoYXQgY29udGFpbnMgdGhlIHBhY2thZ2UuXG4gICAqL1xuICByZWFkb25seSByZXBvc2l0b3J5TmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZm9ybWF0IG9mIHRoZSBwYWNrYWdlIHRoYXQgdHJpZ2dlcmVkIHRoZSBldmVudC5cbiAgICovXG4gIHJlYWRvbmx5IHBhY2thZ2VGb3JtYXQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBvZiB0aGUgcGFja2FnZSB0aGF0IHRyaWdnZXJlZCB0aGUgZXZlbnQuXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlTmFtZXNwYWNlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgcGFja2FnZSB0aGF0IHRyaWdnZXJlZCB0aGUgZXZlbnQuXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgdmVyc2lvbiBvZiB0aGUgcGFja2FnZSB0aGF0IHRyaWdnZXJlZCB0aGUgZXZlbnQuXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlVmVyc2lvbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgc3RhdGUgb2YgdGhlIHBhY2thZ2UgdmVyc2lvbiB3aGVuIHRoZSBldmVudCB3YXMgdHJpZ2dlcmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgcGFja2FnZVZlcnNpb25TdGF0ZTogJ1VuZmluaXNoZWQnIHwgJ1B1Ymxpc2hlZCcgfCAnVW5saXN0ZWQnIHwgJ0FyY2hpdmVkJyB8ICdEaXNwb3NlZCc7XG5cbiAgLyoqXG4gICAqIEEgdmFsdWUgdGhhdCB1bmlxdWVseSBpZGVudGlmaWVzIHRoZSBzdGF0ZSBvZiB0aGUgYXNzZXRzIGFuZCBtZXRhZGF0YSBvZiB0aGUgcGFja2FnZSB2ZXJzaW9uIHdoZW4gdGhlIGV2ZW50IHdhc1xuICAgKiB0cmlnZ2VyZWQuIElmIHRoZSBwYWNrYWdlIHZlcnNpb24gaXMgbW9kaWZpZWQgKGZvciBleGFtcGxlLCBieSBhZGRpbmcgYW5vdGhlciBKQVIgZmlsZSB0byBhIE1hdmVuIHBhY2thZ2UpLCB0aGVcbiAgICogcGFja2FnZVZlcnNpb25SZXZpc2lvbiBjaGFuZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgcGFja2FnZVZlcnNpb25SZXZpc2lvbjogc3RyaW5nO1xuXG4gIHJlYWRvbmx5IGNoYW5nZXM6IHtcbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIGFzc2V0cyBhZGRlZCB0byBhIHBhY2thZ2UgdGhhdCB0cmlnZ2VyZWQgYW4gZXZlbnQuIEV4YW1wbGVzIG9mIGFuIGFzc2V0IGFyZSBhIE1hdmVuIEpBUiBmaWxlIG9yIGFcbiAgICAgKiBQeXRob24gd2hlZWwuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXNzZXRzQWRkZWQ6IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgYXNzZXRzIHJlbW92ZWQgZnJvbSBhIHBhY2thZ2UgdGhhdCB0cmlnZ2VyZWQgYW4gZXZlbnQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXNzZXRzUmVtb3ZlZDogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBhc3NldHMgbW9kaWZpZWQgaW4gdGhlIHBhY2thZ2UgdGhhdCB0cmlnZ2VyZWQgdGhlIGV2ZW50LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1ldGFkYXRhVXBkYXRlZDogYm9vbGVhbjtcblxuICAgIC8qKlxuICAgICAqIEEgYm9vbGVhbiB2YWx1ZSB0aGF0IGlzIHNldCB0byB0cnVlIGlmIHRoZSBldmVudCBpbmNsdWRlcyBtb2RpZmllZCBwYWNrYWdlLWxldmVsIG1ldGFkYXRhLiBGb3IgZXhhbXBsZSwgYW4gZXZlbnRcbiAgICAgKiBtaWdodCBtb2RpZnkgYSBNYXZlbiBwb20ueG1sIGZpbGUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXNzZXRzVXBkYXRlZDogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogQSBib29sZWFuIHZhbHVlIHRoYXQgaXMgc2V0IHRvIHRydWUgaWYgdGhlIGV2ZW50J3MgcGFja2FnZVZlcnNpb25TdGF0dXMgaXMgbW9kaWZpZWQoZm9yIGV4YW1wbGUsIGlmXG4gICAgICogcGFja2FnZVZlcnNpb25TdGF0dXMgY2hhbmdlcyBmcm9tIFVuZmluaXNoZWQgdG8gUHVibGlzaGVkKS5cbiAgICAgKi9cbiAgICByZWFkb25seSBzdGF0dXNDaGFuZ2VkOiBib29sZWFuO1xuICB9O1xuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgdGhlIGhpZ2gtbGV2ZWwgdHlwZSBvZiB0aGUgcGFja2FnZSB2ZXJzaW9uIGNoYW5nZS5cbiAgICovXG4gIHJlYWRvbmx5IG9wZXJhdGlvblR5cGU6ICdDcmVhdGVkJyB8ICdVcGRhdGVkJyB8ICdEZWxldGVkJztcblxuICAvKipcbiAgICogQW4gaW50ZWdlciB0aGF0IHNwZWNpZmllcyBhbiBldmVudCBudW1iZXIgZm9yIGEgcGFja2FnZS4gRWFjaCBldmVudCBvbiBhIHBhY2thZ2UgaW5jcmVtZW50cyB0aGUgc2VxdWVuY2VOdW1iZXIgc29cbiAgICogZXZlbnRzIGNhbiBiZSBhcnJhbmdlZCBzZXF1ZW50aWFsbHkuIEFuIGV2ZW50IGNhbiBpbmNyZW1lbnQgdGhlIHNlcXVlbmNlTnVtYmVyIGJ5IGFueSBpbnRlZ2VyIG51bWJlci5cbiAgICovXG4gIHJlYWRvbmx5IHNlcXVlbmNlTnVtYmVyOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEFuIElEIHVzZWQgdG8gZGlmZmVyZW50aWF0ZSBkdXBsaWNhdGUgRXZlbnRCcmlkZ2UgZXZlbnRzLiBJbiByYXJlIGNhc2VzLCBFdmVudEJyaWRnZSBtaWdodCB0cmlnZ2VyIHRoZSBzYW1lIHJ1bGVcbiAgICogbW9yZSB0aGFuIG9uY2UgZm9yIGEgc2luZ2xlIGV2ZW50IG9yIHNjaGVkdWxlZCB0aW1lLiBPciwgaXQgbWlnaHQgaW52b2tlIHRoZSBzYW1lIHRhcmdldCBtb3JlIHRoYW4gb25jZSBmb3IgYSBnaXZlblxuICAgKiB0cmlnZ2VyZWQgcnVsZS5cbiAgICovXG4gIHJlYWRvbmx5IGV2ZW50RGVkdXBsaWNhdGlvbklkOiBzdHJpbmc7XG59XG4iXX0=