"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'awscdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: constants_lambda_shared_1.MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(constants_lambda_shared_1.MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    ko(new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`));
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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