"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheInvalidator = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const handler_1 = require("./handler");
/**
 * Automatically invalidates paths from a CloudFront distribution based on S3
 * object updates. The invalidated paths are constructed from the S3 object keys
 * being updated, combined with an optional path prefix.
 */
class CacheInvalidator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.pathPrefix != null && !core_1.Token.isUnresolved(props.pathPrefix) && !props.pathPrefix.startsWith('/')) {
            throw new Error('The pathPrefix must start with a leading /');
        }
        const handler = new handler_1.Handler(this, 'Resource', {
            description: `Automated cache invalidation on CloudFront distribution ${props.distribution.distributionId}`,
            environment: {
                DISTRIBUTION_ID: props.distribution.distributionId,
                PATH_PREFIX: (_a = props.pathPrefix) !== null && _a !== void 0 ? _a : '/',
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(1),
        });
        handler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['cloudfront:CreateInvalidation'],
            resources: ['*'],
        }));
        // invalidate cache when a file was created/removed from the bucket
        handler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, {
            events: [aws_s3_1.EventType.OBJECT_CREATED, aws_s3_1.EventType.OBJECT_REMOVED],
        }));
    }
    /**
     * The age of S3 events processed by the cache invalidator. By default this
     * is the MAXIMUM value over a 5 minutes period.
     */
    metricS3EventAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "S3EventAge" /* S3_EVENT_AGE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.CacheInvalidator = CacheInvalidator;
//# sourceMappingURL=data:application/json;base64,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