"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const monitored_certificate_1 = require("../monitored-certificate");
const cache_invalidator_1 = require("./cache-invalidator");
const config_1 = require("./config");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_OPTIMIZED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior('/catalog.json', jsiiObjOrigin, behaviorOptions);
        new cache_invalidator_1.CacheInvalidator(this, 'CacheInvalidator', { bucket: props.packageData, distribution: this.distribution });
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            sources: [s3deploy.Source.asset(webappDir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
        });
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            sources: [s3deploy.Source.asset(config.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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