"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Integrations = exports.LambdaIntegration = exports.Integration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const utils_1 = require("../spec/utils");
/**
 * An integration for an API operation. You can extend this to implement your own integration if you like.
 */
class Integration {
    /**
     * Grant permissions for the API to invoke the integration
     */
    grant(_props) {
        // No permissions to grant by default
    }
}
exports.Integration = Integration;
_a = JSII_RTTI_SYMBOL_1;
Integration[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.Integration", version: "0.12.4" };
/**
 * A lambda integration
 */
class LambdaIntegration extends Integration {
    constructor(lambdaFunction) {
        super();
        this.lambdaFunction = lambdaFunction;
    }
    /**
     * Render the lambda integration as a snippet of OpenAPI
     */
    render(props) {
        return {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri: utils_1.functionInvocationUri(props.scope, this.lambdaFunction),
            passthroughBehavior: "WHEN_NO_MATCH",
        };
    }
    /**
     * Grant API Gateway permissions to invoke the lambda
     */
    grant({ scope, api, operationId, method, path, }) {
        new aws_lambda_1.CfnPermission(scope, `LambdaPermission-${operationId}`, {
            action: "lambda:InvokeFunction",
            principal: "apigateway.amazonaws.com",
            functionName: this.lambdaFunction.functionArn,
            sourceArn: aws_cdk_lib_1.Stack.of(scope).formatArn({
                service: "execute-api",
                resource: api.restApiId,
                // Scope permissions to any stage and a specific method and path of the operation.
                // Path parameters (eg {param} are replaced with wildcards)
                resourceName: `*/${method.toUpperCase()}${path.replace(/{[^\}]*\}/g, "*")}`,
            }),
        });
    }
}
exports.LambdaIntegration = LambdaIntegration;
_b = JSII_RTTI_SYMBOL_1;
LambdaIntegration[_b] = { fqn: "@aws-prototyping-sdk/open-api-gateway.LambdaIntegration", version: "0.12.4" };
/**
 * A collection of integrations to connect API operations with a backend to service requests
 */
class Integrations {
    /**
     * An integration that invokes a lambda function to service the request
     * @param lambdaFunction the function to invoke
     */
    static lambda(lambdaFunction) {
        return new LambdaIntegration(lambdaFunction);
    }
}
exports.Integrations = Integrations;
_c = JSII_RTTI_SYMBOL_1;
Integrations[_c] = { fqn: "@aws-prototyping-sdk/open-api-gateway.Integrations", version: "0.12.4" };
//# sourceMappingURL=data:application/json;base64,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