"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...options?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
        ].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.PYTHON,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated python client is built first
                this.parent.addImplicitDependency(this, this.generatedPythonClient);
            }
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in spec_utils.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "spec_utils.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.10.0" };
//# sourceMappingURL=data:application/json;base64,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