"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNamespaceName = exports.validateArtifactName = exports.validateName = exports.validateSourceAction = exports.validateArtifactBounds = void 0;
const cdk = require("@aws-cdk/core");
const action_1 = require("./action");
/**
 * Validation function that checks if the number of artifacts is within the given bounds
 */
function validateArtifactBounds(type, artifacts, min, max, category, provider) {
    const ret = [];
    if (artifacts.length < min) {
        ret.push(`${category}/${provider} must have at least ${min} ${type} artifacts`);
    }
    if (artifacts.length > max) {
        ret.push(`${category}/${provider} cannot have more than ${max} ${type} artifacts`);
    }
    return ret;
}
exports.validateArtifactBounds = validateArtifactBounds;
/**
 * Validation function that guarantees that an action is or is not a source action. This is useful because Source actions can only be
 * in the first stage of a pipeline, and the first stage can only contain source actions.
 */
function validateSourceAction(mustBeSource, category, actionName, stageName) {
    if (mustBeSource !== (category === action_1.ActionCategory.SOURCE)) {
        return [`Action ${actionName} in stage ${stageName}: ` + (mustBeSource ? 'first stage may only contain Source actions'
                : 'Source actions may only occur in first stage')];
    }
    return [];
}
exports.validateSourceAction = validateSourceAction;
/**
 * Regex to validate Pipeline, Stage, Action names
 *
 * https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html
 */
const VALID_IDENTIFIER_REGEX = /^[a-zA-Z0-9.@_-]{1,100}$/;
/**
 * Validate the given name of a pipeline component. Pipeline component names all have the same restrictions.
 * This can be used to validate the name of all components of a pipeline.
 */
function validateName(thing, name) {
    validateAgainstRegex(VALID_IDENTIFIER_REGEX, thing, name);
}
exports.validateName = validateName;
function validateArtifactName(artifactName) {
    // https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_Artifact.html#CodePipeline-Type-Artifact-name
    validateAgainstRegex(/^[a-zA-Z0-9_-]{1,100}$/, 'Artifact', artifactName);
}
exports.validateArtifactName = validateArtifactName;
function validateNamespaceName(namespaceName) {
    validateAgainstRegex(/^[A-Za-z0-9@_-]{1,100}$/, 'Namespace', namespaceName);
}
exports.validateNamespaceName = validateNamespaceName;
function validateAgainstRegex(regex, thing, name) {
    // name could be a Token - in that case, skip validation altogether
    if (cdk.Token.isUnresolved(name)) {
        return;
    }
    if (name !== undefined && !regex.test(name)) {
        throw new Error(`${thing} name must match regular expression: ${regex.toString()}, got '${name}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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