"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.buildExecutableCommand = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Execute command to run based on package manager configured.
 *
 * @param packageManager package manager being used.
 * @param args args to append.
 */
function buildExecutableCommand(packageManager, ...args) {
    switch (packageManager) {
        case javascript_1.NodePackageManager.YARN:
        case javascript_1.NodePackageManager.YARN2:
            return `yarn ${args.join(" ")}`;
        case javascript_1.NodePackageManager.PNPM:
            return `pnpx ${args.join(" ")}`;
        default:
            return `npx ${args.join(" ")}`;
    }
}
exports.buildExecutableCommand = buildExecutableCommand;
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
            gitignore: [".nx/cache"],
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts"],
            },
        });
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // Add alias task for "projen" to synthesize workspace
        this.addTask("synth-workspace", {
            exec: buildExecutableCommand(this.package.packageManager, "projen"),
            description: "Synthesize workspace",
        });
        this.addTask("run-many", {
            receiveArgs: true,
            exec: buildExecutableCommand(this.package.packageManager, "nx", "run-many"),
            description: "Run task against multiple workspace projects",
        });
        // Map tasks to nx run-many
        if (options.scripts == null || options.scripts.build == null) {
            this._overrideNxBuildTask(this.buildTask, { target: "build" }, true);
        }
        if (options.scripts == null || options.scripts["pre-compile"] == null) {
            this._overrideNxBuildTask(this.preCompileTask, { target: "pre-compile" });
        }
        if (options.scripts == null || options.scripts.compile == null) {
            this._overrideNxBuildTask(this.compileTask, { target: "compile" });
        }
        if (options.scripts == null || options.scripts["post-compile"] == null) {
            this._overrideNxBuildTask(this.postCompileTask, {
                target: "post-compile",
            });
        }
        if (options.scripts == null || options.scripts.test == null) {
            this._overrideNxBuildTask(this.testTask, { target: "test" });
        }
        if (options.scripts == null || options.scripts.eslint == null) {
            this._overrideNxBuildTask(this.eslint?.eslintTask, { target: "eslint" });
        }
        if (options.scripts == null || options.scripts.package == null) {
            this._overrideNxBuildTask(this.packageTask, { target: "package" });
        }
        if (options.scripts == null || options.scripts.prepare == null) {
            this._overrideNxBuildTask("prepare", { target: "prepare" });
        }
        if (options.scripts == null || options.scripts.watch == null) {
            this._overrideNxBuildTask(this.watchTask, {
                target: "watch",
                noBail: false,
                ignoreCycles: true,
                skipCache: true,
                outputStyle: "stream",
            });
        }
        this.addDevDeps("nx", "@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec(buildExecutableCommand(this.package.packageManager, "npm-check-updates", "--deep", "--rejectVersion", "0.0.0", "-u"));
            upgradeDepsTask.exec(buildExecutableCommand(this.package.packageManager, "syncpack", "fix-mismatches"));
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec(buildExecutableCommand(this.package.packageManager, "projen"));
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: options.nxConfig?.cacheableOperations || [
                                "build",
                                "test",
                            ],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                namedInputs: {
                    // https://nx.dev/more-concepts/customizing-inputs#defaults
                    default: ["{projectRoot}/**/*"],
                    ...options.nxConfig?.namedInputs,
                },
                targetDefaults: options.nxConfig?.targetDefaults,
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Helper to format `npx nx run-many ...` style command.
     * @param options
     */
    formatNxRunManyCommand(options) {
        const args = [];
        if (options.configuration) {
            args.push(`--configuration=${options.configuration}`);
        }
        if (options.runner) {
            args.push(`--runner=${options.runner}`);
        }
        if (options.parallel) {
            args.push(`--parallel=${options.parallel}`);
        }
        if (options.skipCache) {
            args.push("--skip-nx-cache");
        }
        if (options.ignoreCycles) {
            args.push("--nx-ignore-cycles");
        }
        if (options.noBail !== true) {
            args.push("--nx-bail");
        }
        if (options.projects && options.projects.length) {
            args.push(`--projects=${options.projects.join(",")}`);
        }
        if (options.exclude) {
            args.push(`--exclude=${options.exclude}`);
        }
        if (options.verbose) {
            args.push("--verbose");
        }
        return buildExecutableCommand(this.package.packageManager, "nx", "run-many", `--target=${options.target}`, `--output-style=${options.outputStyle || "stream"}`, ...args);
    }
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param force - Force unlocking task (eg: build task is locked)
     * @returns - The task that was overridden
     * @internal
     */
    _overrideNxBuildTask(task, options, force) {
        if (typeof task === "string") {
            task = this.tasks.tryFind(task);
        }
        if (task == null) {
            return;
        }
        if (force) {
            // @ts-ignore - private property
            task._locked = false;
        }
        task.reset(this.formatNxRunManyCommand(options), {
            receiveArgs: true,
        });
        task.description += " for all affected projects";
        return task;
    }
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name, options) {
        return this.addTask(name, {
            receiveArgs: true,
            exec: this.formatNxRunManyCommand(options),
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has change, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {
                name: subProject.name,
                private: true,
                __pdk__: true,
                devDependencies: { projen: "*" },
                scripts: subProject.tasks.all.reduce((p, c) => ({
                    [c.name]: `${buildExecutableCommand(this.package.packageManager)} projen ${c.name}`,
                    ...p,
                }), {}),
                version: "0.0.0",
            };
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.disableNoHoistBundled !== true) {
            const noHoistBundled = this.subProjects.flatMap((sub) => {
                if (sub instanceof javascript_1.NodeProject) {
                    return sub.deps.all
                        .filter((dep) => dep.type === projen_1.DependencyType.BUNDLED)
                        .flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        this.package.addField("workspaces", {
            packages: this.workspacePackages,
            nohoist: noHoist,
        });
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`${buildExecutableCommand(this.package.packageManager)} nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.14.21" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibngtbW9ub3JlcG8uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJueC1tb25vcmVwby50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBO3NDQUNzQztBQUN0Qyx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG1DQVNnQjtBQUNoQixzREFJK0I7QUFDL0IsOENBQWtEO0FBQ2xELHNEQUcrQjtBQUUvQix5REFBb0U7QUFFcEUsTUFBTSx1QkFBdUIsR0FBVyxtQ0FBbUMsQ0FBQztBQUU1RTs7Ozs7R0FLRztBQUNILFNBQWdCLHNCQUFzQixDQUNwQyxjQUFrQyxFQUNsQyxHQUFHLElBQWM7SUFFakIsUUFBUSxjQUFjLEVBQUU7UUFDdEIsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLENBQUM7UUFDN0IsS0FBSywrQkFBa0IsQ0FBQyxLQUFLO1lBQzNCLE9BQU8sUUFBUSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7UUFDbEMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJO1lBQzFCLE9BQU8sUUFBUSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7UUFDbEM7WUFDRSxPQUFPLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO0tBQ2xDO0FBQ0gsQ0FBQztBQWJELHdEQWFDO0FBNEhEOzs7OztHQUtHO0FBQ0gsTUFBYSxpQkFBa0IsU0FBUSw4QkFBaUI7SUFXdEQsWUFBWSxPQUFpQztRQUMzQyxLQUFLLENBQUM7WUFDSixHQUFHLE9BQU87WUFDVixNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sSUFBSSxLQUFLO1lBQy9CLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTyxJQUFJLEtBQUs7WUFDakMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRLElBQUksSUFBSTtZQUNsQyxVQUFVLEVBQUUsSUFBSTtZQUNoQixPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU8sSUFBSSxLQUFLO1lBQ2pDLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSSxJQUFJLEtBQUs7WUFDM0Isb0JBQW9CLEVBQUUsT0FBTyxDQUFDLG9CQUFvQixJQUFJLFVBQVU7WUFDaEUsVUFBVSxFQUFFLEtBQUs7WUFDakIsU0FBUyxFQUFFLENBQUMsV0FBVyxDQUFDO1lBQ3hCLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYSxJQUFJO2dCQUN0QyxJQUFJLEVBQUUsQ0FBQyxHQUFHLENBQUM7Z0JBQ1gsY0FBYyxFQUFFLENBQUMsaUJBQWlCLENBQUM7YUFDcEM7WUFDRCxRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVEsSUFBSTtnQkFDNUIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRSxHQUFHO2lCQUNiO2dCQUNELE9BQU8sRUFBRSxDQUFDLFNBQVMsQ0FBQzthQUNyQjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUNqQyxJQUFJLENBQUMsZUFBZSxHQUFHLE9BQU8sQ0FBQyxlQUFlLENBQUM7UUFDL0MsSUFBSSxDQUFDLGlCQUFpQixHQUFHLE9BQU8sQ0FBQyxlQUFlLEVBQUUsa0JBQWtCLElBQUksRUFBRSxDQUFDO1FBQzNFLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLG9CQUFvQixJQUFJLEVBQUUsQ0FBQztRQUV0RSx5Q0FBeUM7UUFDekMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBRXZDLHNEQUFzRDtRQUN0RCxJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFO1lBQzlCLElBQUksRUFBRSxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxRQUFRLENBQUM7WUFDbkUsV0FBVyxFQUFFLHNCQUFzQjtTQUNwQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRTtZQUN2QixXQUFXLEVBQUUsSUFBSTtZQUNqQixJQUFJLEVBQUUsc0JBQXNCLENBQzFCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUMzQixJQUFJLEVBQ0osVUFBVSxDQUNYO1lBQ0QsV0FBVyxFQUFFLDhDQUE4QztTQUM1RCxDQUFDLENBQUM7UUFFSCwyQkFBMkI7UUFDM0IsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssSUFBSSxJQUFJLEVBQUU7WUFDNUQsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsRUFBRSxNQUFNLEVBQUUsT0FBTyxFQUFFLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDdEU7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLElBQUksSUFBSSxFQUFFO1lBQ3JFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFLEVBQUUsTUFBTSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7U0FDM0U7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksRUFBRTtZQUM5RCxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLElBQUksRUFBRTtZQUN0RSxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRTtnQkFDOUMsTUFBTSxFQUFFLGNBQWM7YUFDdkIsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLElBQUksRUFBRTtZQUMzRCxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQzlEO1FBQ0QsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDN0QsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLEVBQUUsTUFBTSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDMUU7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksRUFBRTtZQUM5RCxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxJQUFJLEVBQUU7WUFDOUQsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsRUFBRSxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDO1NBQzdEO1FBQ0QsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssSUFBSSxJQUFJLEVBQUU7WUFDNUQsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7Z0JBQ3hDLE1BQU0sRUFBRSxPQUFPO2dCQUNmLE1BQU0sRUFBRSxLQUFLO2dCQUNiLFlBQVksRUFBRSxJQUFJO2dCQUNsQixTQUFTLEVBQUUsSUFBSTtnQkFDZixXQUFXLEVBQUUsUUFBUTthQUN0QixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLFlBQVksRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDLHVEQUF1RDtRQUM3RyxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLCtCQUErQixDQUFDLENBQUM7UUFFcEUsSUFBSSxPQUFPLENBQUMsbUJBQW1CLEtBQUssS0FBSyxFQUFFO1lBQ3pDLElBQUksQ0FBQyxVQUFVLENBQUMsbUJBQW1CLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFFakQsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDbEMsT0FBTyxDQUFDLDBCQUEwQixFQUFFLFFBQVEsSUFBSSxjQUFjLENBQy9ELENBQUM7WUFDRixlQUFlLENBQUMsSUFBSSxDQUNsQixzQkFBc0IsQ0FDcEIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQzNCLG1CQUFtQixFQUNuQixRQUFRLEVBQ1IsaUJBQWlCLEVBQ2pCLE9BQU8sRUFDUCxJQUFJLENBQ0wsQ0FDRixDQUFDO1lBQ0YsZUFBZSxDQUFDLElBQUksQ0FDbEIsc0JBQXNCLENBQ3BCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUMzQixVQUFVLEVBQ1YsZ0JBQWdCLENBQ2pCLENBQ0YsQ0FBQztZQUNGLGVBQWUsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsVUFBVSxDQUFDLENBQUM7WUFDL0QsZUFBZSxDQUFDLElBQUksQ0FDbEIsc0JBQXNCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsUUFBUSxDQUFDLENBQzlELENBQUM7WUFFRixJQUFJLGlCQUFRLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO2dCQUNyQyxHQUFHLEVBQ0QsT0FBTyxDQUFDLDBCQUEwQixFQUFFLGNBQWMsSUFBSSxpQ0FBYztnQkFDdEUsUUFBUSxFQUFFLElBQUk7YUFDZixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sQ0FBQyxRQUFRLEVBQUUsMEJBQTBCO1lBQzFDLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUVwQyxJQUFJLG1CQUFVLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FDdkMsY0FBYyxFQUNkLFFBQVEsRUFDUixNQUFNLEVBQ04sZUFBZSxFQUNmLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFFBQVEsSUFBSSxFQUFFLENBQUMsQ0FDbkMsQ0FBQztRQUVGLElBQUksaUJBQVEsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7WUFDMUMsUUFBUSxFQUFFLElBQUk7WUFDZCxLQUFLLEVBQUUsRUFBRSxDQUFDLFlBQVksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO1NBQ3RFLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxpQkFBUSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDMUMsR0FBRyxFQUFFO2dCQUNILE9BQU8sRUFBRSxrQ0FBa0M7Z0JBQzNDLE9BQU8sRUFBRSxDQUFDLEtBQUssdUJBQXVCLEVBQUUsQ0FBQztnQkFDekMsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLGtCQUFrQixFQUFFO29CQUNsQixPQUFPLEVBQUU7d0JBQ1AsTUFBTSxFQUFFLE9BQU8sQ0FBQyxRQUFRLEVBQUUsMEJBQTBCOzRCQUNsRCxDQUFDLENBQUMsZ0JBQWdCOzRCQUNsQixDQUFDLENBQUMsdUNBQXVDO3dCQUMzQyxPQUFPLEVBQUU7NEJBQ1AsZ0JBQWdCLEVBQUUsS0FBSzs0QkFDdkIsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLFFBQVEsRUFBRSxtQkFBbUIsSUFBSTtnQ0FDNUQsT0FBTztnQ0FDUCxNQUFNOzZCQUNQOzRCQUNELFdBQVcsRUFBRSxPQUFPLENBQUMsUUFBUSxFQUFFLDBCQUEwQjt5QkFDMUQ7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsV0FBVyxFQUFFO29CQUNYLDJEQUEyRDtvQkFDM0QsT0FBTyxFQUFFLENBQUMsb0JBQW9CLENBQUM7b0JBQy9CLEdBQUcsT0FBTyxDQUFDLFFBQVEsRUFBRSxXQUFXO2lCQUNqQztnQkFDRCxjQUFjLEVBQUUsT0FBTyxDQUFDLFFBQVEsRUFBRSxjQUFjO2dCQUNoRCxvQkFBb0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CO2dCQUMvQyxrQkFBa0IsRUFBRTtvQkFDbEIsS0FBSyxFQUFFO3dCQUNMOzRCQUNFLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxjQUFjO3lCQUN6QjtxQkFDRjtvQkFDRCxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxrQkFBa0IsSUFBSSxFQUFFLENBQUM7aUJBQzdDO2dCQUNELFFBQVEsRUFBRTtvQkFDUixXQUFXLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxjQUFjLElBQUksVUFBVTtpQkFDekQ7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7O09BR0c7SUFDSSxzQkFBc0IsQ0FBQyxPQUF5QjtRQUNyRCxNQUFNLElBQUksR0FBYSxFQUFFLENBQUM7UUFDMUIsSUFBSSxPQUFPLENBQUMsYUFBYSxFQUFFO1lBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLE9BQU8sQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFO1lBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztTQUN6QztRQUNELElBQUksT0FBTyxDQUFDLFFBQVEsRUFBRTtZQUNwQixJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsT0FBTyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDN0M7UUFDRCxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUU7WUFDckIsSUFBSSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQzlCO1FBQ0QsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO1lBQ3hCLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQztTQUNqQztRQUNELElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxJQUFJLEVBQUU7WUFDM0IsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUN4QjtRQUNELElBQUksT0FBTyxDQUFDLFFBQVEsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRTtZQUMvQyxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsSUFBSSxPQUFPLENBQUMsT0FBTyxFQUFFO1lBQ25CLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUMzQztRQUNELElBQUksT0FBTyxDQUFDLE9BQU8sRUFBRTtZQUNuQixJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ3hCO1FBRUQsT0FBTyxzQkFBc0IsQ0FDM0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQzNCLElBQUksRUFDSixVQUFVLEVBQ1YsWUFBWSxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQzVCLGtCQUFrQixPQUFPLENBQUMsV0FBVyxJQUFJLFFBQVEsRUFBRSxFQUNuRCxHQUFHLElBQUksQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDTyxvQkFBb0IsQ0FDNUIsSUFBK0IsRUFDL0IsT0FBeUIsRUFDekIsS0FBZTtRQUVmLElBQUksT0FBTyxJQUFJLEtBQUssUUFBUSxFQUFFO1lBQzVCLElBQUksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNqQztRQUVELElBQUksSUFBSSxJQUFJLElBQUksRUFBRTtZQUNoQixPQUFPO1NBQ1I7UUFFRCxJQUFJLEtBQUssRUFBRTtZQUNULGdDQUFnQztZQUNoQyxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQztTQUN0QjtRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQy9DLFdBQVcsRUFBRSxJQUFJO1NBQ2xCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxXQUFXLElBQUksNEJBQTRCLENBQUM7UUFFakQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxnQkFBZ0IsQ0FBQyxJQUFZLEVBQUUsT0FBeUI7UUFDN0QsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRTtZQUN4QixXQUFXLEVBQUUsSUFBSTtZQUNqQixJQUFJLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLE9BQU8sQ0FBQztTQUMzQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLHFCQUFxQixDQUFDLFNBQWtCLEVBQUUsUUFBaUI7UUFDaEUsSUFBSSxJQUFJLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQzdDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMvRDthQUFNO1lBQ0wsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUM3RDtJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxvQkFBb0IsQ0FBQyxHQUFHLFlBQXNCO1FBQ25ELGdIQUFnSDtRQUNoSCxpREFBaUQ7UUFDakQsTUFBTSxtQ0FBbUMsR0FDdkMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLEdBQUcsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQ2pELElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLENBQzNDLENBQUM7UUFDSixNQUFNLHlCQUF5QixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQ3pCLEdBQUcsbUNBQW1DLENBQUMsTUFBTSxDQUMzQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyx5QkFBeUIsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQzdDLENBQ0YsQ0FBQztRQUVGLG1DQUFtQztRQUNuQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLEdBQUcsWUFBWSxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVELDZEQUE2RDtJQUM3RCxJQUFZLDZCQUE2QjtRQUN2QyxhQUFhO1FBQ2IsTUFBTSxXQUFXLEdBQWMsSUFBSSxDQUFDLFdBQVcsSUFBSSxFQUFFLENBQUM7UUFDdEQsT0FBTyxXQUFXLENBQUM7SUFDckIsQ0FBQztJQUVELElBQVcsV0FBVztRQUNwQixPQUFPLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FDdEQsQ0FBQyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUM3QixDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0gsS0FBSztRQUNILElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBQzNCLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztRQUN2QixJQUFJLENBQUMsc0JBQXNCLEVBQUUsQ0FBQztRQUM5QixJQUFJLENBQUMsNEJBQTRCLEVBQUUsQ0FBQztRQUVwQyx5R0FBeUc7UUFDekcsNEZBQTRGO1FBQzVGLE1BQU0sa0JBQWtCLEdBQWtCLEVBQUUsQ0FBQztRQUM3QyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFO1lBQ3RDLElBQUksYUFBYSxDQUFDLFVBQVUsQ0FBQyxFQUFFO2dCQUM3QixNQUFNLGNBQWMsR0FBZ0IsVUFBeUIsQ0FBQztnQkFDOUQsa0JBQWtCLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDaEQsZ0RBQWdEO2dCQUNoRCxjQUFjLENBQUMsT0FBTyxDQUFDLG1CQUFtQixHQUFHLEdBQUcsRUFBRSxHQUFFLENBQUMsQ0FBQzthQUN2RDtRQUNILENBQUMsQ0FBQyxDQUFDO1FBRUgsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWQsdUdBQXVHO1FBQ3ZHO1FBQ0UsaUNBQWlDO1FBQ2hDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBaUIsQ0FBQyxPQUFPLEtBQUssSUFBSTtZQUNoRCxpQ0FBaUM7WUFDakMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBRSxHQUFHLENBQUMsSUFBaUIsQ0FBQyxPQUFPLEtBQUssSUFBSSxDQUFDLEVBQ3pFO1lBQ0EsZ0RBQWdEO1lBQ2hELElBQUksQ0FBQyxPQUFPLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztTQUNwQztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNLLG1CQUFtQjtRQUN6QixJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQWUsRUFBRSxFQUFFO1lBQzNDLDRFQUE0RTtZQUM1RSxVQUFVLENBQUMsV0FBVyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBRWhDLElBQ0UsYUFBYSxDQUFDLFVBQVUsQ0FBQztnQkFDekIsVUFBVSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQ2pFO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsR0FBRyxVQUFVLENBQUMsSUFBSSwrREFBK0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEdBQUcsQ0FDaEgsQ0FBQzthQUNIO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSyw0QkFBNEI7UUFDbEMsSUFBSSxDQUFDLFdBQVc7YUFDYixNQUFNLENBQUMsQ0FBQyxVQUFlLEVBQUUsRUFBRSxDQUFDLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDO2FBQ3ZELE1BQU0sQ0FBQyxDQUFDLFVBQW1CLEVBQUUsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQzthQUN4RSxPQUFPLENBQUMsQ0FBQyxVQUFtQixFQUFFLEVBQUU7WUFDL0IsdUNBQXVDO1lBQ3ZDLE1BQU0sUUFBUSxHQUFRO2dCQUNwQixJQUFJLEVBQUUsVUFBVSxDQUFDLElBQUk7Z0JBQ3JCLE9BQU8sRUFBRSxJQUFJO2dCQUNiLE9BQU8sRUFBRSxJQUFJO2dCQUNiLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRSxHQUFHLEVBQUU7Z0JBQ2hDLE9BQU8sRUFBRSxVQUFVLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQ2xDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztvQkFDVCxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxHQUFHLHNCQUFzQixDQUNqQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FDNUIsV0FBVyxDQUFDLENBQUMsSUFBSSxFQUFFO29CQUNwQixHQUFHLENBQUM7aUJBQ0wsQ0FBQyxFQUNGLEVBQUUsQ0FDSDtnQkFDRCxPQUFPLEVBQUUsT0FBTzthQUNqQixDQUFDO1lBRUYsSUFBSSxpQkFBUSxDQUFDLFVBQVUsRUFBRSxjQUFjLEVBQUU7Z0JBQ3ZDLEdBQUcsRUFBRSxRQUFRO2dCQUNiLFFBQVEsRUFBRSxJQUFJO2FBQ2YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxlQUFlO1FBQ3JCLGlIQUFpSDtRQUNqSCwyQ0FBMkM7UUFDM0MsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7UUFFNUIsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLGVBQWUsRUFBRSxPQUFPLENBQUM7UUFDNUMsZ0lBQWdJO1FBQ2hJLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRSxxQkFBcUIsS0FBSyxJQUFJLEVBQUU7WUFDeEQsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtnQkFDdEQsSUFBSSxHQUFHLFlBQVksd0JBQVcsRUFBRTtvQkFDOUIsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUc7eUJBQ2hCLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyx1QkFBYyxDQUFDLE9BQU8sQ0FBQzt5QkFDcEQsT0FBTyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQzt3QkFDaEIsR0FBRyxHQUFHLENBQUMsSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLEVBQUU7d0JBQ3pCLEdBQUcsR0FBRyxDQUFDLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxJQUFJO3FCQUM1QixDQUFDLENBQUM7aUJBQ047Z0JBQ0QsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztZQUVILElBQUksY0FBYyxDQUFDLE1BQU0sRUFBRTtnQkFDekIsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUMsRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO2FBQ25EO1NBQ0Y7UUFFRCxxQ0FBcUM7UUFDckMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7WUFDM0QsSUFBSSxpQkFBUSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtnQkFDeEMsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsR0FBRyxFQUFFO29CQUNILFFBQVEsRUFBRSxJQUFJLENBQUMsaUJBQWlCO2lCQUNqQzthQUNGLENBQUMsQ0FBQztTQUNKO1FBQ0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFO1lBQ2xDLFFBQVEsRUFBRSxJQUFJLENBQUMsaUJBQWlCO1lBQ2hDLE9BQU8sRUFBRSxPQUFPO1NBQ2pCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssc0JBQXNCO1FBQzVCLDJCQUEyQjtRQUMzQixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FDNUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLE9BQU8sWUFBWSxzQkFBYSxDQUMzQixDQUFDO1FBRXJCLElBQUksY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDL0IsdUNBQXVDO1lBQ3ZDLE9BQU87U0FDUjtRQUVELHNIQUFzSDtRQUN0SCxpSEFBaUg7UUFDakgsY0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFO1lBQ3ZDLE1BQU0sYUFBYSxHQUNqQixhQUFhLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUM7Z0JBQ3pDLGFBQWEsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDdEMsTUFBTSxXQUFXLEdBQUcsYUFBYSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7WUFFM0QsQ0FBQyxXQUFXLEVBQUUsS0FBSyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO2dCQUMxQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxhQUFhLENBQUMsQ0FBQztZQUM1RCxDQUFDLENBQUMsQ0FBQztZQUVILFdBQVcsRUFBRSxLQUFLLEVBQUUsQ0FBQztRQUN2QixDQUFDLENBQUMsQ0FBQztRQUVILHFIQUFxSDtRQUNySCxrR0FBa0c7UUFDbEcsTUFBTSxtQkFBbUIsR0FDdkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUMzRCxtQkFBbUIsQ0FBQyxJQUFJLENBQ3RCLEdBQUcsc0JBQXNCLENBQ3ZCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUM1QiwrQ0FBK0MsY0FBYzthQUMzRCxHQUFHLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUM7YUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQzVCLENBQUM7UUFFRix3RUFBd0U7UUFDeEUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsK0JBQStCLEVBQUU7WUFDdkQ7Z0JBQ0UsTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLFFBQVEsRUFBRSxjQUFjO2FBQ3pCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7T0FHRztJQUNLLGdCQUFnQixDQUFDLElBQWMsRUFBRSxJQUFVLEVBQUUsT0FBZ0I7UUFDbkUsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ2IsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQzFEO2FBQU0sSUFBSSxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztTQUN4RDthQUFNLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNyQixNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdEQsSUFBSSxXQUFXLEVBQUU7Z0JBQ2YsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7YUFDN0Q7U0FDRjthQUFNLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUN2QixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUM1QjtJQUNILENBQUM7O0FBcmhCSCw4Q0FzaEJDOzs7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsYUFBYSxDQUFDLE9BQVk7SUFDakMsT0FBTyxPQUFPLFlBQVksd0JBQVcsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDO0FBQzNELENBQUM7QUFFRCxTQUFTLGFBQWE7SUFDcEIsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSw4QkFBOEIsQ0FBQyxDQUFDO0FBQzlELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIGZzIGZyb20gXCJmc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHtcbiAgRGVwZW5kZW5jeVR5cGUsXG4gIElnbm9yZUZpbGUsXG4gIEpzb25GaWxlLFxuICBQcm9qZWN0LFxuICBUYXNrLFxuICBUYXNrU3RlcCxcbiAgVGV4dEZpbGUsXG4gIFlhbWxGaWxlLFxufSBmcm9tIFwicHJvamVuXCI7XG5pbXBvcnQge1xuICBOb2RlUGFja2FnZSxcbiAgTm9kZVBhY2thZ2VNYW5hZ2VyLFxuICBOb2RlUHJvamVjdCxcbn0gZnJvbSBcInByb2plbi9saWIvamF2YXNjcmlwdFwiO1xuaW1wb3J0IHsgUHl0aG9uUHJvamVjdCB9IGZyb20gXCJwcm9qZW4vbGliL3B5dGhvblwiO1xuaW1wb3J0IHtcbiAgVHlwZVNjcmlwdFByb2plY3QsXG4gIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyxcbn0gZnJvbSBcInByb2plbi9saWIvdHlwZXNjcmlwdFwiO1xuaW1wb3J0IHsgTnggfSBmcm9tIFwiLi9ueC10eXBlc1wiO1xuaW1wb3J0IHsgREVGQVVMVF9DT05GSUcsIFN5bmNwYWNrQ29uZmlnIH0gZnJvbSBcIi4vc3luY3BhY2stb3B0aW9uc1wiO1xuXG5jb25zdCBOWF9NT05PUkVQT19QTFVHSU5fUEFUSDogc3RyaW5nID0gXCIubngvcGx1Z2lucy9ueC1tb25vcmVwby1wbHVnaW4uanNcIjtcblxuLyoqXG4gKiBFeGVjdXRlIGNvbW1hbmQgdG8gcnVuIGJhc2VkIG9uIHBhY2thZ2UgbWFuYWdlciBjb25maWd1cmVkLlxuICpcbiAqIEBwYXJhbSBwYWNrYWdlTWFuYWdlciBwYWNrYWdlIG1hbmFnZXIgYmVpbmcgdXNlZC5cbiAqIEBwYXJhbSBhcmdzIGFyZ3MgdG8gYXBwZW5kLlxuICovXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRFeGVjdXRhYmxlQ29tbWFuZChcbiAgcGFja2FnZU1hbmFnZXI6IE5vZGVQYWNrYWdlTWFuYWdlcixcbiAgLi4uYXJnczogc3RyaW5nW11cbikge1xuICBzd2l0Y2ggKHBhY2thZ2VNYW5hZ2VyKSB7XG4gICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuWUFSTjpcbiAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOMjpcbiAgICAgIHJldHVybiBgeWFybiAke2FyZ3Muam9pbihcIiBcIil9YDtcbiAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNOlxuICAgICAgcmV0dXJuIGBwbnB4ICR7YXJncy5qb2luKFwiIFwiKX1gO1xuICAgIGRlZmF1bHQ6XG4gICAgICByZXR1cm4gYG5weCAke2FyZ3Muam9pbihcIiBcIil9YDtcbiAgfVxufVxuXG4vKipcbiAqIFdvcmtzcGFjZSBjb25maWd1cmF0aW9ucy5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vY2xhc3NpYy55YXJucGtnLmNvbS9sYW5nL2VuL2RvY3Mvd29ya3NwYWNlcy9cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXb3Jrc3BhY2VDb25maWcge1xuICAvKipcbiAgICogTGlzdCBvZiBwYWNrYWdlIGdsb2JzIHRvIGV4Y2x1ZGUgZnJvbSBob2lzdGluZyBpbiB0aGUgd29ya3NwYWNlLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vY2xhc3NpYy55YXJucGtnLmNvbS9ibG9nLzIwMTgvMDIvMTUvbm9ob2lzdC9cbiAgICovXG4gIHJlYWRvbmx5IG5vSG9pc3Q/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogRGlzYWJsZSBhdXRvbWF0aWNhbGx5IGFwcGx5aW5nIGBub0hvaXN0YCBsb2dpYyBmb3IgYWxsIHN1Yi1wcm9qZWN0IFwiYnVuZGxlZERlcGVuZGVuY2llc1wiLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGlzYWJsZU5vSG9pc3RCdW5kbGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTGlzdCBvZiBhZGRpdGlvbmFsIHBhY2thZ2UgZ2xvYnMgdG8gaW5jbHVkZSBpbiB0aGUgd29ya3NwYWNlLlxuICAgKlxuICAgKiBBbGwgcGFja2FnZXMgd2hpY2ggYXJlIHBhcmVudGVkIGJ5IHRoZSBtb25vcmVwbyBhcmUgYXV0b21hdGljYWxseSBhZGRlZCB0byB0aGUgd29ya3NwYWNlLCBidXQgeW91IGNhbiB1c2UgdGhpc1xuICAgKiBwcm9wZXJ0eSB0byBzcGVjaWZ5IGFueSBhZGRpdGlvbmFsIHBhdGhzIHRvIHBhY2thZ2VzIHdoaWNoIG1heSBub3QgYmUgbWFuYWdlZCBieSBwcm9qZW4uXG4gICAqL1xuICByZWFkb25seSBhZGRpdGlvbmFsUGFja2FnZXM/OiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBNb25vcmVwbyBVcGdyYWRlIERlcHMgdGFzay5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNb25vcmVwb1VwZ3JhZGVEZXBzT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSB0YXNrIHRvIGNyZWF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdXBncmFkZS1kZXBzXG4gICAqL1xuICByZWFkb25seSB0YXNrTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogU3luY3BhY2sgY29uZmlndXJhdGlvbi5cbiAgICpcbiAgICogTm8gbWVyZ2luZyBpcyBwZXJmb3JtZWQgYW5kIGFzIHN1Y2ggYSBjb21wbGV0ZSBzeW5jcGFja0NvbmZpZyBpcyByZXF1aXJlZCBpZiBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgU3luY3BhY2tDb25maWcuREVGQVVMVF9DT05GSUdcbiAgICovXG4gIHJlYWRvbmx5IHN5bmNwYWNrQ29uZmlnPzogU3luY3BhY2tDb25maWc7XG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBvcHRpb25zIGZvciB0aGUgTnhNb25vcmVwb1Byb2plY3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTnhNb25vcmVwb1Byb2plY3RPcHRpb25zIGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIENvbmZpZ3VyYXRpb24gZm9yIE5YLlxuICAgKi9cbiAgcmVhZG9ubHkgbnhDb25maWc/OiBOeC5Xb3Jrc3BhY2VDb25maWc7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyYXRpb24gZm9yIHdvcmtzcGFjZS5cbiAgICovXG4gIHJlYWRvbmx5IHdvcmtzcGFjZUNvbmZpZz86IFdvcmtzcGFjZUNvbmZpZztcblxuICAvKipcbiAgICogV2hldGhlciB0byBpbmNsdWRlIGFuIHVwZ3JhZGUtZGVwcyB0YXNrIGF0IHRoZSByb290IG9mIHRoZSBtb25vcmVwbyB3aGljaCB3aWxsIHVwZ3JhZGUgYWxsIGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbW9ub3JlcG9VcGdyYWRlRGVwcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE1vbm9yZXBvIFVwZ3JhZGUgRGVwcyBvcHRpb25zLlxuICAgKlxuICAgKiBUaGlzIGlzIG9ubHkgdXNlZCBpZiBtb25vcmVwb1VwZ3JhZGVEZXBzIGlzIHRydWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgbW9ub3JlcG9VcGdyYWRlRGVwc09wdGlvbnM/OiBNb25vcmVwb1VwZ3JhZGVEZXBzT3B0aW9ucztcbn1cblxuLyoqXG4gKlxuICogQHNlZSBodHRwczovL254LmRldi9wYWNrYWdlcy9ueC9kb2N1bWVudHMvcnVuLW1hbnkjb3B0aW9uc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIE54UnVuTWFueU9wdGlvbnMge1xuICAvKiogVGFzayB0byBydW4gZm9yIGFmZmVjdGVkIHByb2plY3RzICovXG4gIHJlYWRvbmx5IHRhcmdldDogc3RyaW5nO1xuICAvKiogVGhpcyBpcyB0aGUgY29uZmlndXJhdGlvbiB0byB1c2Ugd2hlbiBwZXJmb3JtaW5nIHRhc2tzIG9uIHByb2plY3RzICovXG4gIHJlYWRvbmx5IGNvbmZpZ3VyYXRpb24/OiBzdHJpbmc7XG4gIC8qKiBFeGNsdWRlIGNlcnRhaW4gcHJvamVjdHMgZnJvbSBiZWluZyBwcm9jZXNzZWQgKi9cbiAgcmVhZG9ubHkgZXhjbHVkZT86IHN0cmluZztcbiAgLyoqXG4gICAqIERvIG5vdCBzdG9wIGNvbW1hbmQgZXhlY3V0aW9uIGFmdGVyIHRoZSBmaXJzdCBmYWlsZWQgdGFzay5cbiAgICovXG4gIHJlYWRvbmx5IG5vQmFpbD86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBEZWZpbmVzIGhvdyBOeCBlbWl0cyBvdXRwdXRzIHRhc2tzIGxvZ3NcbiAgICogQGRlZmF1bHQgXCJzdHJlYW1cIlxuICAgKi9cbiAgcmVhZG9ubHkgb3V0cHV0U3R5bGU/OlxuICAgIHwgXCJkeW5hbWljXCJcbiAgICB8IFwic3RhdGljXCJcbiAgICB8IFwic3RyZWFtXCJcbiAgICB8IFwic3RyZWFtLXdpdGhvdXQtcHJlZml4ZXNcIjtcbiAgLyoqXG4gICAqIE1heCBudW1iZXIgb2YgcGFyYWxsZWwgcHJvY2Vzc2VzXG4gICAqIEBkZWZhdWx0IDNcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFsbGVsPzogbnVtYmVyO1xuICAvKiogUHJvamVjdCB0byBydW4gYXMgbGlzdCBwcm9qZWN0IG5hbWVzIGFuZC9vciBwYXR0ZXJucy4gKi9cbiAgcmVhZG9ubHkgcHJvamVjdHM/OiBzdHJpbmdbXTtcbiAgLyoqIFRoaXMgaXMgdGhlIG5hbWUgb2YgdGhlIHRhc2tzIHJ1bm5lciBjb25maWd1cmF0aW9uIGluIG54Lmpzb24gKi9cbiAgcmVhZG9ubHkgcnVubmVyPzogc3RyaW5nO1xuICAvKiogUmVydW4gdGhlIHRhc2tzIGV2ZW4gd2hlbiB0aGUgcmVzdWx0cyBhcmUgYXZhaWxhYmxlIGluIHRoZSBjYWNoZS4gKi9cbiAgcmVhZG9ubHkgc2tpcENhY2hlPzogYm9vbGVhbjtcbiAgLyoqIElnbm9yZSBjeWNsZXMgaW4gdGhlIHRhc2sgZ3JhcGggKi9cbiAgcmVhZG9ubHkgaWdub3JlQ3ljbGVzPzogYm9vbGVhbjtcbiAgLyoqIFByaW50cyBhZGRpdGlvbmFsIGluZm9ybWF0aW9uIGFib3V0IHRoZSBjb21tYW5kcyAoZS5nLiBzdGFjayB0cmFjZXMpLiAqL1xuICByZWFkb25seSB2ZXJib3NlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBUaGlzIHByb2plY3QgdHlwZSB3aWxsIGJvb3RzdHJhcCBhIE5YIGJhc2VkIG1vbm9yZXBvIHdpdGggc3VwcG9ydCBmb3IgcG9seWdvdFxuICogYnVpbGRzLCBidWlsZCBjYWNoaW5nLCBkZXBlbmRlbmN5IGdyYXBoIHZpc3VhbGl6YXRpb24gYW5kIG11Y2ggbW9yZS5cbiAqXG4gKiBAcGppZCBueC1tb25vcmVwb1xuICovXG5leHBvcnQgY2xhc3MgTnhNb25vcmVwb1Byb2plY3QgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdCB7XG4gIC8vIG11dGFibGUgZGF0YSBzdHJ1Y3R1cmVzXG4gIHByaXZhdGUgcmVhZG9ubHkgaW1wbGljaXREZXBlbmRlbmNpZXM6IE54LkltcGxpY2l0RGVwZW5kZW5jaWVzO1xuXG4gIC8vIGltbXV0YWJsZSBkYXRhIHN0cnVjdHVyZXNcbiAgcHJpdmF0ZSByZWFkb25seSBueENvbmZpZz86IE54LldvcmtzcGFjZUNvbmZpZztcbiAgcHJpdmF0ZSByZWFkb25seSB3b3Jrc3BhY2VDb25maWc/OiBXb3Jrc3BhY2VDb25maWc7XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya3NwYWNlUGFja2FnZXM6IHN0cmluZ1tdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgbnhKc29uOiBKc29uRmlsZTtcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBOeE1vbm9yZXBvUHJvamVjdE9wdGlvbnMpIHtcbiAgICBzdXBlcih7XG4gICAgICAuLi5vcHRpb25zLFxuICAgICAgZ2l0aHViOiBvcHRpb25zLmdpdGh1YiA/PyBmYWxzZSxcbiAgICAgIHBhY2thZ2U6IG9wdGlvbnMucGFja2FnZSA/PyBmYWxzZSxcbiAgICAgIHByZXR0aWVyOiBvcHRpb25zLnByZXR0aWVyID8/IHRydWUsXG4gICAgICBwcm9qZW5yY1RzOiB0cnVlLFxuICAgICAgcmVsZWFzZTogb3B0aW9ucy5yZWxlYXNlID8/IGZhbHNlLFxuICAgICAgamVzdDogb3B0aW9ucy5qZXN0ID8/IGZhbHNlLFxuICAgICAgZGVmYXVsdFJlbGVhc2VCcmFuY2g6IG9wdGlvbnMuZGVmYXVsdFJlbGVhc2VCcmFuY2ggPz8gXCJtYWlubGluZVwiLFxuICAgICAgc2FtcGxlQ29kZTogZmFsc2UsIC8vIHJvb3Qgc2hvdWxkIG5ldmVyIGhhdmUgc2FtcGxlIGNvZGUsXG4gICAgICBnaXRpZ25vcmU6IFtcIi5ueC9jYWNoZVwiXSxcbiAgICAgIGVzbGludE9wdGlvbnM6IG9wdGlvbnMuZXNsaW50T3B0aW9ucyA/PyB7XG4gICAgICAgIGRpcnM6IFtcIi5cIl0sXG4gICAgICAgIGlnbm9yZVBhdHRlcm5zOiBbXCJwYWNrYWdlcy8qKi8qLipcIl0sXG4gICAgICB9LFxuICAgICAgdHNjb25maWc6IG9wdGlvbnMudHNjb25maWcgPz8ge1xuICAgICAgICBjb21waWxlck9wdGlvbnM6IHtcbiAgICAgICAgICByb290RGlyOiBcIi5cIixcbiAgICAgICAgfSxcbiAgICAgICAgaW5jbHVkZTogW1wiKiovKi50c1wiXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLm54Q29uZmlnID0gb3B0aW9ucy5ueENvbmZpZztcbiAgICB0aGlzLndvcmtzcGFjZUNvbmZpZyA9IG9wdGlvbnMud29ya3NwYWNlQ29uZmlnO1xuICAgIHRoaXMud29ya3NwYWNlUGFja2FnZXMgPSBvcHRpb25zLndvcmtzcGFjZUNvbmZpZz8uYWRkaXRpb25hbFBhY2thZ2VzID8/IFtdO1xuICAgIHRoaXMuaW1wbGljaXREZXBlbmRlbmNpZXMgPSB0aGlzLm54Q29uZmlnPy5pbXBsaWNpdERlcGVuZGVuY2llcyB8fCB7fTtcblxuICAgIC8vIE5ldmVyIHB1Ymxpc2ggYSBtb25vcmVwbyByb290IHBhY2thZ2UuXG4gICAgdGhpcy5wYWNrYWdlLmFkZEZpZWxkKFwicHJpdmF0ZVwiLCB0cnVlKTtcblxuICAgIC8vIEFkZCBhbGlhcyB0YXNrIGZvciBcInByb2plblwiIHRvIHN5bnRoZXNpemUgd29ya3NwYWNlXG4gICAgdGhpcy5hZGRUYXNrKFwic3ludGgtd29ya3NwYWNlXCIsIHtcbiAgICAgIGV4ZWM6IGJ1aWxkRXhlY3V0YWJsZUNvbW1hbmQodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyLCBcInByb2plblwiKSxcbiAgICAgIGRlc2NyaXB0aW9uOiBcIlN5bnRoZXNpemUgd29ya3NwYWNlXCIsXG4gICAgfSk7XG5cbiAgICB0aGlzLmFkZFRhc2soXCJydW4tbWFueVwiLCB7XG4gICAgICByZWNlaXZlQXJnczogdHJ1ZSxcbiAgICAgIGV4ZWM6IGJ1aWxkRXhlY3V0YWJsZUNvbW1hbmQoXG4gICAgICAgIHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcixcbiAgICAgICAgXCJueFwiLFxuICAgICAgICBcInJ1bi1tYW55XCJcbiAgICAgICksXG4gICAgICBkZXNjcmlwdGlvbjogXCJSdW4gdGFzayBhZ2FpbnN0IG11bHRpcGxlIHdvcmtzcGFjZSBwcm9qZWN0c1wiLFxuICAgIH0pO1xuXG4gICAgLy8gTWFwIHRhc2tzIHRvIG54IHJ1bi1tYW55XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy5idWlsZCA9PSBudWxsKSB7XG4gICAgICB0aGlzLl9vdmVycmlkZU54QnVpbGRUYXNrKHRoaXMuYnVpbGRUYXNrLCB7IHRhcmdldDogXCJidWlsZFwiIH0sIHRydWUpO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5zY3JpcHRzID09IG51bGwgfHwgb3B0aW9ucy5zY3JpcHRzW1wicHJlLWNvbXBpbGVcIl0gPT0gbnVsbCkge1xuICAgICAgdGhpcy5fb3ZlcnJpZGVOeEJ1aWxkVGFzayh0aGlzLnByZUNvbXBpbGVUYXNrLCB7IHRhcmdldDogXCJwcmUtY29tcGlsZVwiIH0pO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5zY3JpcHRzID09IG51bGwgfHwgb3B0aW9ucy5zY3JpcHRzLmNvbXBpbGUgPT0gbnVsbCkge1xuICAgICAgdGhpcy5fb3ZlcnJpZGVOeEJ1aWxkVGFzayh0aGlzLmNvbXBpbGVUYXNrLCB7IHRhcmdldDogXCJjb21waWxlXCIgfSk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnNjcmlwdHMgPT0gbnVsbCB8fCBvcHRpb25zLnNjcmlwdHNbXCJwb3N0LWNvbXBpbGVcIl0gPT0gbnVsbCkge1xuICAgICAgdGhpcy5fb3ZlcnJpZGVOeEJ1aWxkVGFzayh0aGlzLnBvc3RDb21waWxlVGFzaywge1xuICAgICAgICB0YXJnZXQ6IFwicG9zdC1jb21waWxlXCIsXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy50ZXN0ID09IG51bGwpIHtcbiAgICAgIHRoaXMuX292ZXJyaWRlTnhCdWlsZFRhc2sodGhpcy50ZXN0VGFzaywgeyB0YXJnZXQ6IFwidGVzdFwiIH0pO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5zY3JpcHRzID09IG51bGwgfHwgb3B0aW9ucy5zY3JpcHRzLmVzbGludCA9PSBudWxsKSB7XG4gICAgICB0aGlzLl9vdmVycmlkZU54QnVpbGRUYXNrKHRoaXMuZXNsaW50Py5lc2xpbnRUYXNrLCB7IHRhcmdldDogXCJlc2xpbnRcIiB9KTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy5wYWNrYWdlID09IG51bGwpIHtcbiAgICAgIHRoaXMuX292ZXJyaWRlTnhCdWlsZFRhc2sodGhpcy5wYWNrYWdlVGFzaywgeyB0YXJnZXQ6IFwicGFja2FnZVwiIH0pO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5zY3JpcHRzID09IG51bGwgfHwgb3B0aW9ucy5zY3JpcHRzLnByZXBhcmUgPT0gbnVsbCkge1xuICAgICAgdGhpcy5fb3ZlcnJpZGVOeEJ1aWxkVGFzayhcInByZXBhcmVcIiwgeyB0YXJnZXQ6IFwicHJlcGFyZVwiIH0pO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5zY3JpcHRzID09IG51bGwgfHwgb3B0aW9ucy5zY3JpcHRzLndhdGNoID09IG51bGwpIHtcbiAgICAgIHRoaXMuX292ZXJyaWRlTnhCdWlsZFRhc2sodGhpcy53YXRjaFRhc2ssIHtcbiAgICAgICAgdGFyZ2V0OiBcIndhdGNoXCIsXG4gICAgICAgIG5vQmFpbDogZmFsc2UsXG4gICAgICAgIGlnbm9yZUN5Y2xlczogdHJ1ZSxcbiAgICAgICAgc2tpcENhY2hlOiB0cnVlLFxuICAgICAgICBvdXRwdXRTdHlsZTogXCJzdHJlYW1cIixcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuYWRkRGV2RGVwcyhcIm54XCIsIFwiQG5yd2wvY2xpXCIsIFwiQG5yd2wvd29ya3NwYWNlXCIpO1xuICAgIHRoaXMuYWRkRGVwcyhcImF3cy1jZGstbGliXCIsIFwiY29uc3RydWN0c1wiLCBcImNkay1uYWdcIik7IC8vIE5lZWRlZCBhcyB0aGlzIGNhbiBiZSBidW5kbGVkIGluIGF3cy1wcm90b3R5cGluZy1zZGtcbiAgICB0aGlzLnBhY2thZ2UuYWRkUGFja2FnZVJlc29sdXRpb25zKFwiQHR5cGVzL2JhYmVsX190cmF2ZXJzZUA3LjE4LjJcIik7XG5cbiAgICBpZiAob3B0aW9ucy5tb25vcmVwb1VwZ3JhZGVEZXBzICE9PSBmYWxzZSkge1xuICAgICAgdGhpcy5hZGREZXZEZXBzKFwibnBtLWNoZWNrLXVwZGF0ZXNcIiwgXCJzeW5jcGFja1wiKTtcblxuICAgICAgY29uc3QgdXBncmFkZURlcHNUYXNrID0gdGhpcy5hZGRUYXNrKFxuICAgICAgICBvcHRpb25zLm1vbm9yZXBvVXBncmFkZURlcHNPcHRpb25zPy50YXNrTmFtZSB8fCBcInVwZ3JhZGUtZGVwc1wiXG4gICAgICApO1xuICAgICAgdXBncmFkZURlcHNUYXNrLmV4ZWMoXG4gICAgICAgIGJ1aWxkRXhlY3V0YWJsZUNvbW1hbmQoXG4gICAgICAgICAgdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyLFxuICAgICAgICAgIFwibnBtLWNoZWNrLXVwZGF0ZXNcIixcbiAgICAgICAgICBcIi0tZGVlcFwiLFxuICAgICAgICAgIFwiLS1yZWplY3RWZXJzaW9uXCIsXG4gICAgICAgICAgXCIwLjAuMFwiLFxuICAgICAgICAgIFwiLXVcIlxuICAgICAgICApXG4gICAgICApO1xuICAgICAgdXBncmFkZURlcHNUYXNrLmV4ZWMoXG4gICAgICAgIGJ1aWxkRXhlY3V0YWJsZUNvbW1hbmQoXG4gICAgICAgICAgdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyLFxuICAgICAgICAgIFwic3luY3BhY2tcIixcbiAgICAgICAgICBcImZpeC1taXNtYXRjaGVzXCJcbiAgICAgICAgKVxuICAgICAgKTtcbiAgICAgIHVwZ3JhZGVEZXBzVGFzay5leGVjKGAke3RoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcn0gaW5zdGFsbGApO1xuICAgICAgdXBncmFkZURlcHNUYXNrLmV4ZWMoXG4gICAgICAgIGJ1aWxkRXhlY3V0YWJsZUNvbW1hbmQodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyLCBcInByb2plblwiKVxuICAgICAgKTtcblxuICAgICAgbmV3IEpzb25GaWxlKHRoaXMsIFwiLnN5bmNwYWNrcmMuanNvblwiLCB7XG4gICAgICAgIG9iajpcbiAgICAgICAgICBvcHRpb25zLm1vbm9yZXBvVXBncmFkZURlcHNPcHRpb25zPy5zeW5jcGFja0NvbmZpZyB8fCBERUZBVUxUX0NPTkZJRyxcbiAgICAgICAgcmVhZG9ubHk6IHRydWUsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBvcHRpb25zLm54Q29uZmlnPy5ueENsb3VkUmVhZE9ubHlBY2Nlc3NUb2tlbiAmJlxuICAgICAgdGhpcy5hZGREZXZEZXBzKFwiQG5yd2wvbngtY2xvdWRcIik7XG5cbiAgICBuZXcgSWdub3JlRmlsZSh0aGlzLCBcIi5ueGlnbm9yZVwiKS5leGNsdWRlKFxuICAgICAgXCJ0ZXN0LXJlcG9ydHNcIixcbiAgICAgIFwidGFyZ2V0XCIsXG4gICAgICBcIi5lbnZcIixcbiAgICAgIFwiLnB5dGVzdF9jYWNoZVwiLFxuICAgICAgLi4uKHRoaXMubnhDb25maWc/Lm54SWdub3JlIHx8IFtdKVxuICAgICk7XG5cbiAgICBuZXcgVGV4dEZpbGUodGhpcywgTlhfTU9OT1JFUE9fUExVR0lOX1BBVEgsIHtcbiAgICAgIHJlYWRvbmx5OiB0cnVlLFxuICAgICAgbGluZXM6IGZzLnJlYWRGaWxlU3luYyhnZXRQbHVnaW5QYXRoKCkpLnRvU3RyaW5nKFwidXRmLThcIikuc3BsaXQoXCJcXG5cIiksXG4gICAgfSk7XG5cbiAgICB0aGlzLm54SnNvbiA9IG5ldyBKc29uRmlsZSh0aGlzLCBcIm54Lmpzb25cIiwge1xuICAgICAgb2JqOiB7XG4gICAgICAgIGV4dGVuZHM6IFwiQG5yd2wvd29ya3NwYWNlL3ByZXNldHMvbnBtLmpzb25cIixcbiAgICAgICAgcGx1Z2luczogW2AuLyR7TlhfTU9OT1JFUE9fUExVR0lOX1BBVEh9YF0sXG4gICAgICAgIG5wbVNjb3BlOiBcIm1vbm9yZXBvXCIsXG4gICAgICAgIHRhc2tzUnVubmVyT3B0aW9uczoge1xuICAgICAgICAgIGRlZmF1bHQ6IHtcbiAgICAgICAgICAgIHJ1bm5lcjogb3B0aW9ucy5ueENvbmZpZz8ubnhDbG91ZFJlYWRPbmx5QWNjZXNzVG9rZW5cbiAgICAgICAgICAgICAgPyBcIkBucndsL254LWNsb3VkXCJcbiAgICAgICAgICAgICAgOiBcIkBucndsL3dvcmtzcGFjZS90YXNrcy1ydW5uZXJzL2RlZmF1bHRcIixcbiAgICAgICAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgICAgICAgdXNlRGFlbW9uUHJvY2VzczogZmFsc2UsXG4gICAgICAgICAgICAgIGNhY2hlYWJsZU9wZXJhdGlvbnM6IG9wdGlvbnMubnhDb25maWc/LmNhY2hlYWJsZU9wZXJhdGlvbnMgfHwgW1xuICAgICAgICAgICAgICAgIFwiYnVpbGRcIixcbiAgICAgICAgICAgICAgICBcInRlc3RcIixcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgYWNjZXNzVG9rZW46IG9wdGlvbnMubnhDb25maWc/Lm54Q2xvdWRSZWFkT25seUFjY2Vzc1Rva2VuLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBuYW1lZElucHV0czoge1xuICAgICAgICAgIC8vIGh0dHBzOi8vbnguZGV2L21vcmUtY29uY2VwdHMvY3VzdG9taXppbmctaW5wdXRzI2RlZmF1bHRzXG4gICAgICAgICAgZGVmYXVsdDogW1wie3Byb2plY3RSb290fS8qKi8qXCJdLFxuICAgICAgICAgIC4uLm9wdGlvbnMubnhDb25maWc/Lm5hbWVkSW5wdXRzLFxuICAgICAgICB9LFxuICAgICAgICB0YXJnZXREZWZhdWx0czogb3B0aW9ucy5ueENvbmZpZz8udGFyZ2V0RGVmYXVsdHMsXG4gICAgICAgIGltcGxpY2l0RGVwZW5kZW5jaWVzOiB0aGlzLmltcGxpY2l0RGVwZW5kZW5jaWVzLFxuICAgICAgICB0YXJnZXREZXBlbmRlbmNpZXM6IHtcbiAgICAgICAgICBidWlsZDogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICB0YXJnZXQ6IFwiYnVpbGRcIixcbiAgICAgICAgICAgICAgcHJvamVjdHM6IFwiZGVwZW5kZW5jaWVzXCIsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgLi4uKHRoaXMubnhDb25maWc/LnRhcmdldERlcGVuZGVuY2llcyB8fCB7fSksXG4gICAgICAgIH0sXG4gICAgICAgIGFmZmVjdGVkOiB7XG4gICAgICAgICAgZGVmYXVsdEJhc2U6IHRoaXMubnhDb25maWc/LmFmZmVjdGVkQnJhbmNoIHx8IFwibWFpbmxpbmVcIixcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogSGVscGVyIHRvIGZvcm1hdCBgbnB4IG54IHJ1bi1tYW55IC4uLmAgc3R5bGUgY29tbWFuZC5cbiAgICogQHBhcmFtIG9wdGlvbnNcbiAgICovXG4gIHB1YmxpYyBmb3JtYXROeFJ1bk1hbnlDb21tYW5kKG9wdGlvbnM6IE54UnVuTWFueU9wdGlvbnMpOiBzdHJpbmcge1xuICAgIGNvbnN0IGFyZ3M6IHN0cmluZ1tdID0gW107XG4gICAgaWYgKG9wdGlvbnMuY29uZmlndXJhdGlvbikge1xuICAgICAgYXJncy5wdXNoKGAtLWNvbmZpZ3VyYXRpb249JHtvcHRpb25zLmNvbmZpZ3VyYXRpb259YCk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnJ1bm5lcikge1xuICAgICAgYXJncy5wdXNoKGAtLXJ1bm5lcj0ke29wdGlvbnMucnVubmVyfWApO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5wYXJhbGxlbCkge1xuICAgICAgYXJncy5wdXNoKGAtLXBhcmFsbGVsPSR7b3B0aW9ucy5wYXJhbGxlbH1gKTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2tpcENhY2hlKSB7XG4gICAgICBhcmdzLnB1c2goXCItLXNraXAtbngtY2FjaGVcIik7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLmlnbm9yZUN5Y2xlcykge1xuICAgICAgYXJncy5wdXNoKFwiLS1ueC1pZ25vcmUtY3ljbGVzXCIpO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy5ub0JhaWwgIT09IHRydWUpIHtcbiAgICAgIGFyZ3MucHVzaChcIi0tbngtYmFpbFwiKTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMucHJvamVjdHMgJiYgb3B0aW9ucy5wcm9qZWN0cy5sZW5ndGgpIHtcbiAgICAgIGFyZ3MucHVzaChgLS1wcm9qZWN0cz0ke29wdGlvbnMucHJvamVjdHMuam9pbihcIixcIil9YCk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLmV4Y2x1ZGUpIHtcbiAgICAgIGFyZ3MucHVzaChgLS1leGNsdWRlPSR7b3B0aW9ucy5leGNsdWRlfWApO1xuICAgIH1cbiAgICBpZiAob3B0aW9ucy52ZXJib3NlKSB7XG4gICAgICBhcmdzLnB1c2goXCItLXZlcmJvc2VcIik7XG4gICAgfVxuXG4gICAgcmV0dXJuIGJ1aWxkRXhlY3V0YWJsZUNvbW1hbmQoXG4gICAgICB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIsXG4gICAgICBcIm54XCIsXG4gICAgICBcInJ1bi1tYW55XCIsXG4gICAgICBgLS10YXJnZXQ9JHtvcHRpb25zLnRhcmdldH1gLFxuICAgICAgYC0tb3V0cHV0LXN0eWxlPSR7b3B0aW9ucy5vdXRwdXRTdHlsZSB8fCBcInN0cmVhbVwifWAsXG4gICAgICAuLi5hcmdzXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBPdmVycmlkZXMgXCJidWlsZFwiIHJlbGF0ZWQgcHJvamVjdCB0YXNrcyAoYnVpbGQsIGNvbXBpbGUsIHRlc3QsIGV0Yy4pIHdpdGggYG5weCBueCBydW4tbWFueWAgZm9ybWF0LlxuICAgKiBAcGFyYW0gdGFzayAtIFRoZSB0YXNrIG9yIHRhc2sgbmFtZSB0byBvdmVycmlkZVxuICAgKiBAcGFyYW0gb3B0aW9ucyAtIE54IHJ1bi1tYW55IG9wdGlvbnNcbiAgICogQHBhcmFtIGZvcmNlIC0gRm9yY2UgdW5sb2NraW5nIHRhc2sgKGVnOiBidWlsZCB0YXNrIGlzIGxvY2tlZClcbiAgICogQHJldHVybnMgLSBUaGUgdGFzayB0aGF0IHdhcyBvdmVycmlkZGVuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHJvdGVjdGVkIF9vdmVycmlkZU54QnVpbGRUYXNrKFxuICAgIHRhc2s6IFRhc2sgfCBzdHJpbmcgfCB1bmRlZmluZWQsXG4gICAgb3B0aW9uczogTnhSdW5NYW55T3B0aW9ucyxcbiAgICBmb3JjZT86IGJvb2xlYW5cbiAgKTogVGFzayB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHR5cGVvZiB0YXNrID09PSBcInN0cmluZ1wiKSB7XG4gICAgICB0YXNrID0gdGhpcy50YXNrcy50cnlGaW5kKHRhc2spO1xuICAgIH1cblxuICAgIGlmICh0YXNrID09IG51bGwpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAoZm9yY2UpIHtcbiAgICAgIC8vIEB0cy1pZ25vcmUgLSBwcml2YXRlIHByb3BlcnR5XG4gICAgICB0YXNrLl9sb2NrZWQgPSBmYWxzZTtcbiAgICB9XG5cbiAgICB0YXNrLnJlc2V0KHRoaXMuZm9ybWF0TnhSdW5NYW55Q29tbWFuZChvcHRpb25zKSwge1xuICAgICAgcmVjZWl2ZUFyZ3M6IHRydWUsXG4gICAgfSk7XG5cbiAgICB0YXNrLmRlc2NyaXB0aW9uICs9IFwiIGZvciBhbGwgYWZmZWN0ZWQgcHJvamVjdHNcIjtcblxuICAgIHJldHVybiB0YXNrO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBwcm9qZWN0IHRhc2sgdGhhdCBleGVjdXRlcyBgbnB4IG54IHJ1bi1tYW55IC4uLmAgc3R5bGUgY29tbWFuZC5cbiAgICovXG4gIHB1YmxpYyBhZGROeFJ1bk1hbnlUYXNrKG5hbWU6IHN0cmluZywgb3B0aW9uczogTnhSdW5NYW55T3B0aW9ucyk6IFRhc2sge1xuICAgIHJldHVybiB0aGlzLmFkZFRhc2sobmFtZSwge1xuICAgICAgcmVjZWl2ZUFyZ3M6IHRydWUsXG4gICAgICBleGVjOiB0aGlzLmZvcm1hdE54UnVuTWFueUNvbW1hbmQob3B0aW9ucyksXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGFuIGltcGxpY2l0IGRlcGVuZGVuY3kgYmV0d2VlbiB0d28gUHJvamVjdCdzLiBUaGlzIGlzIHR5cGljYWxseVxuICAgKiB1c2VkIGluIHBvbHlnb3QgcmVwb3Mgd2hlcmUgYSBUeXBlc2NyaXB0IHByb2plY3Qgd2FudHMgYSBidWlsZCBkZXBlbmRlbmN5XG4gICAqIG9uIGEgUHl0aG9uIHByb2plY3QgYXMgYW4gZXhhbXBsZS5cbiAgICpcbiAgICogQHBhcmFtIGRlcGVuZGVudCBwcm9qZWN0IHlvdSB3YW50IHRvIGhhdmUgdGhlIGRlcGVuZGVuY3kuXG4gICAqIEBwYXJhbSBkZXBlbmRlZSBwcm9qZWN0IHlvdSB3aXNoIHRvIGRlcGVuZCBvbi5cbiAgICovXG4gIHB1YmxpYyBhZGRJbXBsaWNpdERlcGVuZGVuY3koZGVwZW5kZW50OiBQcm9qZWN0LCBkZXBlbmRlZTogUHJvamVjdCkge1xuICAgIGlmICh0aGlzLmltcGxpY2l0RGVwZW5kZW5jaWVzW2RlcGVuZGVudC5uYW1lXSkge1xuICAgICAgdGhpcy5pbXBsaWNpdERlcGVuZGVuY2llc1tkZXBlbmRlbnQubmFtZV0ucHVzaChkZXBlbmRlZS5uYW1lKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5pbXBsaWNpdERlcGVuZGVuY2llc1tkZXBlbmRlbnQubmFtZV0gPSBbZGVwZW5kZWUubmFtZV07XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBvbmUgb3IgbW9yZSBhZGRpdGlvbmFsIHBhY2thZ2UgZ2xvYnMgdG8gdGhlIHdvcmtzcGFjZS5cbiAgICogQHBhcmFtIHBhY2thZ2VHbG9icyBwYXRocyB0byB0aGUgcGFja2FnZSB0byBpbmNsdWRlIGluIHRoZSB3b3Jrc3BhY2UgKGZvciBleGFtcGxlIHBhY2thZ2VzL215LXBhY2thZ2UpXG4gICAqL1xuICBwdWJsaWMgYWRkV29ya3NwYWNlUGFja2FnZXMoLi4ucGFja2FnZUdsb2JzOiBzdHJpbmdbXSkge1xuICAgIC8vIEFueSBzdWJwcm9qZWN0cyB0aGF0IHdlcmUgYWRkZWQgc2luY2UgdGhlIGxhc3QgY2FsbCB0byB0aGlzIG1ldGhvZCBuZWVkIHRvIGJlIGFkZGVkIGZpcnN0LCBpbiBvcmRlciB0byBlbnN1cmVcbiAgICAvLyB3ZSBhZGQgdGhlIHdvcmtzcGFjZSBwYWNrYWdlcyBpbiBhIHNhbmUgb3JkZXIuXG4gICAgY29uc3QgcmVsYXRpdmVTdWJQcm9qZWN0V29ya3NwYWNlUGFja2FnZXMgPVxuICAgICAgdGhpcy5pbnN0YW50aWF0aW9uT3JkZXJTdWJQcm9qZWN0cy5tYXAoKHByb2plY3QpID0+XG4gICAgICAgIHBhdGgucmVsYXRpdmUodGhpcy5vdXRkaXIsIHByb2plY3Qub3V0ZGlyKVxuICAgICAgKTtcbiAgICBjb25zdCBleGlzdGluZ1dvcmtzcGFjZVBhY2thZ2VzID0gbmV3IFNldCh0aGlzLndvcmtzcGFjZVBhY2thZ2VzKTtcbiAgICB0aGlzLndvcmtzcGFjZVBhY2thZ2VzLnB1c2goXG4gICAgICAuLi5yZWxhdGl2ZVN1YlByb2plY3RXb3Jrc3BhY2VQYWNrYWdlcy5maWx0ZXIoXG4gICAgICAgIChwa2cpID0+ICFleGlzdGluZ1dvcmtzcGFjZVBhY2thZ2VzLmhhcyhwa2cpXG4gICAgICApXG4gICAgKTtcblxuICAgIC8vIEFkZCB0aGUgYWRkaXRpb25hbCBwYWNrYWdlcyBuZXh0XG4gICAgdGhpcy53b3Jrc3BhY2VQYWNrYWdlcy5wdXNoKC4uLnBhY2thZ2VHbG9icyk7XG4gIH1cblxuICAvLyBSZW1vdmUgdGhpcyBoYWNrIG9uY2Ugc3ViUHJvamVjdHMgaXMgbWFkZSBwdWJsaWMgaW4gUHJvamVuXG4gIHByaXZhdGUgZ2V0IGluc3RhbnRpYXRpb25PcmRlclN1YlByb2plY3RzKCk6IFByb2plY3RbXSB7XG4gICAgLy8gQHRzLWlnbm9yZVxuICAgIGNvbnN0IHN1YlByb2plY3RzOiBQcm9qZWN0W10gPSB0aGlzLnN1YnByb2plY3RzIHx8IFtdO1xuICAgIHJldHVybiBzdWJQcm9qZWN0cztcbiAgfVxuXG4gIHB1YmxpYyBnZXQgc3ViUHJvamVjdHMoKTogUHJvamVjdFtdIHtcbiAgICByZXR1cm4gdGhpcy5pbnN0YW50aWF0aW9uT3JkZXJTdWJQcm9qZWN0cy5zb3J0KChhLCBiKSA9PlxuICAgICAgYS5uYW1lLmxvY2FsZUNvbXBhcmUoYi5uYW1lKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQGluaGVyaXREb2NcbiAgICovXG4gIHN5bnRoKCkge1xuICAgIHRoaXMudmFsaWRhdGVTdWJQcm9qZWN0cygpO1xuICAgIHRoaXMudXBkYXRlV29ya3NwYWNlKCk7XG4gICAgdGhpcy53aXJlUHl0aG9uRGVwZW5kZW5jaWVzKCk7XG4gICAgdGhpcy5zeW50aGVzaXplTm9uTm9kZVBhY2thZ2VKc29uKCk7XG5cbiAgICAvLyBQcmV2ZW50IHN1YiBOb2RlUHJvamVjdCBwYWNrYWdlcyBmcm9tIGBwb3N0U3ludGhlc2lzYCB3aGljaCB3aWxsIGNhdXNlIGluZGl2aWR1YWwvZXh0cmFuZW91cyBpbnN0YWxscy5cbiAgICAvLyBUaGUgd29ya3NwYWNlIHBhY2thZ2UgaW5zdGFsbCB3aWxsIGhhbmRsZSBhbGwgdGhlIHN1YiBOb2RlUHJvamVjdCBwYWNrYWdlcyBhdXRvbWF0aWNhbGx5LlxuICAgIGNvbnN0IHN1YlByb2plY3RQYWNrYWdlczogTm9kZVBhY2thZ2VbXSA9IFtdO1xuICAgIHRoaXMuc3ViUHJvamVjdHMuZm9yRWFjaCgoc3ViUHJvamVjdCkgPT4ge1xuICAgICAgaWYgKGlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkpIHtcbiAgICAgICAgY29uc3Qgc3ViTm9kZVByb2plY3Q6IE5vZGVQcm9qZWN0ID0gc3ViUHJvamVjdCBhcyBOb2RlUHJvamVjdDtcbiAgICAgICAgc3ViUHJvamVjdFBhY2thZ2VzLnB1c2goc3ViTm9kZVByb2plY3QucGFja2FnZSk7XG4gICAgICAgIC8vIEB0cy1pZ25vcmUgLSBgaW5zdGFsbERlcGVuZGVuY2llc2AgaXMgcHJpdmF0ZVxuICAgICAgICBzdWJOb2RlUHJvamVjdC5wYWNrYWdlLmluc3RhbGxEZXBlbmRlbmNpZXMgPSAoKSA9PiB7fTtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHN1cGVyLnN5bnRoKCk7XG5cbiAgICAvLyBGb3JjZSB3b3Jrc3BhY2UgaW5zdGFsbCBkZXBzIGlmIGFueSBub2RlIHN1YnByb2plY3QgcGFja2FnZSBoYXMgY2hhbmdlLCB1bmxlc3MgdGhlIHdvcmtzcGFjZSBjaGFuZ2VkXG4gICAgaWYgKFxuICAgICAgLy8gQHRzLWlnbm9yZSAtIGBmaWxlYCBpcyBwcml2YXRlXG4gICAgICAodGhpcy5wYWNrYWdlLmZpbGUgYXMgSnNvbkZpbGUpLmNoYW5nZWQgIT09IHRydWUgJiZcbiAgICAgIC8vIEB0cy1pZ25vcmUgLSBgZmlsZWAgaXMgcHJpdmF0ZVxuICAgICAgc3ViUHJvamVjdFBhY2thZ2VzLmZpbmQoKHBrZykgPT4gKHBrZy5maWxlIGFzIEpzb25GaWxlKS5jaGFuZ2VkID09PSB0cnVlKVxuICAgICkge1xuICAgICAgLy8gQHRzLWlnbm9yZSAtIGBpbnN0YWxsRGVwZW5kZW5jaWVzYCBpcyBwcml2YXRlXG4gICAgICB0aGlzLnBhY2thZ2UuaW5zdGFsbERlcGVuZGVuY2llcygpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBFbnN1cmVzIHN1YnByb2plY3RzIGRvbid0IGhhdmUgYSBkZWZhdWx0IHRhc2sgYW5kIHRoYXQgYWxsIHBhY2thZ2VzIHVzZSB0aGUgc2FtZSBwYWNrYWdlIG1hbmFnZXIuXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlU3ViUHJvamVjdHMoKSB7XG4gICAgdGhpcy5zdWJQcm9qZWN0cy5mb3JFYWNoKChzdWJQcm9qZWN0OiBhbnkpID0+IHtcbiAgICAgIC8vIERpc2FibGUgZGVmYXVsdCB0YXNrIG9uIHN1YnByb2plY3RzIGFzIHRoaXMgaXNuJ3Qgc3VwcG9ydGVkIGluIGEgbW9ub3JlcG9cbiAgICAgIHN1YlByb2plY3QuZGVmYXVsdFRhc2s/LnJlc2V0KCk7XG5cbiAgICAgIGlmIChcbiAgICAgICAgaXNOb2RlUHJvamVjdChzdWJQcm9qZWN0KSAmJlxuICAgICAgICBzdWJQcm9qZWN0LnBhY2thZ2UucGFja2FnZU1hbmFnZXIgIT09IHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlclxuICAgICAgKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgJHtzdWJQcm9qZWN0Lm5hbWV9IHBhY2thZ2VNYW5hZ2VyIGRvZXMgbm90IG1hdGNoIHRoZSBtb25vcmVwbyBwYWNrYWdlTWFuYWdlcjogJHt0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJ9LmBcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGb3Igbm9uLW5vZGUgcHJvamVjdHMsIGEgcGFja2FnZS5qc29uIGlzIHJlcXVpcmVkIGluIG9yZGVyIHRvIGJlIGRpc2NvdmVyZWQgYnkgTlguXG4gICAqL1xuICBwcml2YXRlIHN5bnRoZXNpemVOb25Ob2RlUGFja2FnZUpzb24oKSB7XG4gICAgdGhpcy5zdWJQcm9qZWN0c1xuICAgICAgLmZpbHRlcigoc3ViUHJvamVjdDogYW55KSA9PiAhaXNOb2RlUHJvamVjdChzdWJQcm9qZWN0KSlcbiAgICAgIC5maWx0ZXIoKHN1YlByb2plY3Q6IFByb2plY3QpID0+ICFzdWJQcm9qZWN0LnRyeUZpbmRGaWxlKFwicGFja2FnZS5qc29uXCIpKVxuICAgICAgLmZvckVhY2goKHN1YlByb2plY3Q6IFByb2plY3QpID0+IHtcbiAgICAgICAgLy8gZ2VuZXJhdGUgYSBwYWNrYWdlLmpzb24gaWYgbm90IGZvdW5kXG4gICAgICAgIGNvbnN0IG1hbmlmZXN0OiBhbnkgPSB7XG4gICAgICAgICAgbmFtZTogc3ViUHJvamVjdC5uYW1lLFxuICAgICAgICAgIHByaXZhdGU6IHRydWUsXG4gICAgICAgICAgX19wZGtfXzogdHJ1ZSxcbiAgICAgICAgICBkZXZEZXBlbmRlbmNpZXM6IHsgcHJvamVuOiBcIipcIiB9LFxuICAgICAgICAgIHNjcmlwdHM6IHN1YlByb2plY3QudGFza3MuYWxsLnJlZHVjZShcbiAgICAgICAgICAgIChwLCBjKSA9PiAoe1xuICAgICAgICAgICAgICBbYy5uYW1lXTogYCR7YnVpbGRFeGVjdXRhYmxlQ29tbWFuZChcbiAgICAgICAgICAgICAgICB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJcbiAgICAgICAgICAgICAgKX0gcHJvamVuICR7Yy5uYW1lfWAsXG4gICAgICAgICAgICAgIC4uLnAsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIHt9XG4gICAgICAgICAgKSxcbiAgICAgICAgICB2ZXJzaW9uOiBcIjAuMC4wXCIsXG4gICAgICAgIH07XG5cbiAgICAgICAgbmV3IEpzb25GaWxlKHN1YlByb2plY3QsIFwicGFja2FnZS5qc29uXCIsIHtcbiAgICAgICAgICBvYmo6IG1hbmlmZXN0LFxuICAgICAgICAgIHJlYWRvbmx5OiB0cnVlLFxuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHN1Ym1vZHVsZSBlbnRyeSB0byB0aGUgYXBwcm9wcmlhdGUgd29ya3NwYWNlIGZpbGUuXG4gICAqL1xuICBwcml2YXRlIHVwZGF0ZVdvcmtzcGFjZSgpIHtcbiAgICAvLyBBIGZpbmFsIGNhbGwgdG8gYWRkV29ya3NwYWNlUGFja2FnZXMgd2lsbCB1cGRhdGUgdGhlIGxpc3Qgb2Ygd29ya3NwYWNlIHBhY2thZ2VzIHdpdGggYW55IHN1YnByb2plY3RzIHRoYXQgaGF2ZVxuICAgIC8vIG5vdCB5ZXQgYmVlbiBhZGRlZCwgaW4gdGhlIGNvcnJlY3Qgb3JkZXJcbiAgICB0aGlzLmFkZFdvcmtzcGFjZVBhY2thZ2VzKCk7XG5cbiAgICBsZXQgbm9Ib2lzdCA9IHRoaXMud29ya3NwYWNlQ29uZmlnPy5ub0hvaXN0O1xuICAgIC8vIEF1dG9tYXRpY2FsbHkgYWRkIGFsbCBzdWItcHJvamVjdCBcImJ1bmRsZWREZXBlbmRlbmNpZXNcIiB0byB3b3Jrc3BhY2UgXCJob2hvaXN0XCIsIG90aGVyd2lzZSB0aGV5IGFyZSBub3QgYnVuZGxlZCBpbiBucG0gcGFja2FnZVxuICAgIGlmICh0aGlzLndvcmtzcGFjZUNvbmZpZz8uZGlzYWJsZU5vSG9pc3RCdW5kbGVkICE9PSB0cnVlKSB7XG4gICAgICBjb25zdCBub0hvaXN0QnVuZGxlZCA9IHRoaXMuc3ViUHJvamVjdHMuZmxhdE1hcCgoc3ViKSA9PiB7XG4gICAgICAgIGlmIChzdWIgaW5zdGFuY2VvZiBOb2RlUHJvamVjdCkge1xuICAgICAgICAgIHJldHVybiBzdWIuZGVwcy5hbGxcbiAgICAgICAgICAgIC5maWx0ZXIoKGRlcCkgPT4gZGVwLnR5cGUgPT09IERlcGVuZGVuY3lUeXBlLkJVTkRMRUQpXG4gICAgICAgICAgICAuZmxhdE1hcCgoZGVwKSA9PiBbXG4gICAgICAgICAgICAgIGAke3N1Yi5uYW1lfS8ke2RlcC5uYW1lfWAsXG4gICAgICAgICAgICAgIGAke3N1Yi5uYW1lfS8ke2RlcC5uYW1lfS8qYCxcbiAgICAgICAgICAgIF0pO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICAgIH0pO1xuXG4gICAgICBpZiAobm9Ib2lzdEJ1bmRsZWQubGVuZ3RoKSB7XG4gICAgICAgIG5vSG9pc3QgPSBbLi4uKG5vSG9pc3QgfHwgW10pLCAuLi5ub0hvaXN0QnVuZGxlZF07XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gQWRkIHdvcmtzcGFjZXMgZm9yIGVhY2ggc3VicHJvamVjdFxuICAgIGlmICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNKSB7XG4gICAgICBuZXcgWWFtbEZpbGUodGhpcywgXCJwbnBtLXdvcmtzcGFjZS55YW1sXCIsIHtcbiAgICAgICAgcmVhZG9ubHk6IHRydWUsXG4gICAgICAgIG9iajoge1xuICAgICAgICAgIHBhY2thZ2VzOiB0aGlzLndvcmtzcGFjZVBhY2thZ2VzLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICAgIHRoaXMucGFja2FnZS5hZGRGaWVsZChcIndvcmtzcGFjZXNcIiwge1xuICAgICAgcGFja2FnZXM6IHRoaXMud29ya3NwYWNlUGFja2FnZXMsXG4gICAgICBub2hvaXN0OiBub0hvaXN0LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIGluc3RhbGwgdGFzayBmb3IgcHl0aG9uIHByb2plY3RzIHNvIHRoYXQgdGhleSBhcmUgcnVuIHNlcmlhbGx5IGFuZCBpbiBkZXBlbmRlbmN5IG9yZGVyIHN1Y2ggdGhhdCBweXRob25cbiAgICogcHJvamVjdHMgd2l0aGluIHRoZSBtb25vcmVwbyBjYW4gZGVjbGFyZSBkZXBlbmRlbmNpZXMgb24gb25lIGFub3RoZXIuXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIHdpcmVQeXRob25EZXBlbmRlbmNpZXMoKSB7XG4gICAgLy8gRmluZCBhbnkgcHl0aG9uIHByb2plY3RzXG4gICAgY29uc3QgcHl0aG9uUHJvamVjdHMgPSB0aGlzLnN1YlByb2plY3RzLmZpbHRlcihcbiAgICAgIChwcm9qZWN0KSA9PiBwcm9qZWN0IGluc3RhbmNlb2YgUHl0aG9uUHJvamVjdFxuICAgICkgYXMgUHl0aG9uUHJvamVjdFtdO1xuXG4gICAgaWYgKHB5dGhvblByb2plY3RzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgLy8gTm90aGluZyB0byBkbyBmb3Igbm8gcHl0aG9uIHByb2plY3RzXG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgLy8gTW92ZSBhbGwgaW5zdGFsbCBjb21tYW5kcyB0byBpbnN0YWxsLXB5IHNvIHRoYXQgdGhleSBhcmUgbm90IGluc3RhbGxlZCBpbiBwYXJhbGxlbCBieSB0aGUgbW9ub3JlcG8gcGFja2FnZSBtYW5hZ2VyLlxuICAgIC8vIGVnIHlhcm4gaW5zdGFsbCB3aWxsIHJ1biB0aGUgaW5zdGFsbCB0YXNrIGZvciBhbGwgcGFja2FnZXMgaW4gcGFyYWxsZWwgd2hpY2ggY2FuIGxlYWQgdG8gY29uZmxpY3RzIGZvciBweXRob24uXG4gICAgcHl0aG9uUHJvamVjdHMuZm9yRWFjaCgocHl0aG9uUHJvamVjdCkgPT4ge1xuICAgICAgY29uc3QgaW5zdGFsbFB5VGFzayA9XG4gICAgICAgIHB5dGhvblByb2plY3QudGFza3MudHJ5RmluZChcImluc3RhbGwtcHlcIikgPz9cbiAgICAgICAgcHl0aG9uUHJvamVjdC5hZGRUYXNrKFwiaW5zdGFsbC1weVwiKTtcbiAgICAgIGNvbnN0IGluc3RhbGxUYXNrID0gcHl0aG9uUHJvamVjdC50YXNrcy50cnlGaW5kKFwiaW5zdGFsbFwiKTtcblxuICAgICAgKGluc3RhbGxUYXNrPy5zdGVwcyB8fCBbXSkuZm9yRWFjaCgoc3RlcCkgPT4ge1xuICAgICAgICB0aGlzLmNvcHlTdGVwSW50b1Rhc2soc3RlcCwgaW5zdGFsbFB5VGFzaywgcHl0aG9uUHJvamVjdCk7XG4gICAgICB9KTtcblxuICAgICAgaW5zdGFsbFRhc2s/LnJlc2V0KCk7XG4gICAgfSk7XG5cbiAgICAvLyBBZGQgYW4gaW5zdGFsbCB0YXNrIHRvIHRoZSBtb25vcmVwbyB0byBpbmNsdWRlIHJ1bm5pbmcgdGhlIGluc3RhbGwtcHkgY29tbWFuZCBzZXJpYWxseSB0byBhdm9pZCBjb25mbGljdGluZyB3cml0ZXNcbiAgICAvLyB0byBhIHNoYXJlZCB2aXJ0dWFsIGVudi4gVGhpcyBpcyBhbHNvIG1hbmFnZWQgYnkgbnggc28gdGhhdCBpbnN0YWxscyBvY2N1ciBpbiBkZXBlbmRlbmN5IG9yZGVyLlxuICAgIGNvbnN0IG1vbm9yZXBvSW5zdGFsbFRhc2sgPVxuICAgICAgdGhpcy50YXNrcy50cnlGaW5kKFwiaW5zdGFsbFwiKSA/PyB0aGlzLmFkZFRhc2soXCJpbnN0YWxsXCIpO1xuICAgIG1vbm9yZXBvSW5zdGFsbFRhc2suZXhlYyhcbiAgICAgIGAke2J1aWxkRXhlY3V0YWJsZUNvbW1hbmQoXG4gICAgICAgIHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlclxuICAgICAgKX0gbnggcnVuLW1hbnkgLS10YXJnZXQgaW5zdGFsbC1weSAtLXByb2plY3RzICR7cHl0aG9uUHJvamVjdHNcbiAgICAgICAgLm1hcCgocHJvamVjdCkgPT4gcHJvamVjdC5uYW1lKVxuICAgICAgICAuam9pbihcIixcIil9IC0tcGFyYWxsZWw9MWBcbiAgICApO1xuXG4gICAgLy8gVXBkYXRlIHRoZSBueC5qc29uIHRvIGVuc3VyZSB0aGF0IGluc3RhbGwtcHkgZm9sbG93cyBkZXBlbmRlbmN5IG9yZGVyXG4gICAgdGhpcy5ueEpzb24uYWRkT3ZlcnJpZGUoXCJ0YXJnZXREZXBlbmRlbmNpZXMuaW5zdGFsbC1weVwiLCBbXG4gICAgICB7XG4gICAgICAgIHRhcmdldDogXCJpbnN0YWxsLXB5XCIsXG4gICAgICAgIHByb2plY3RzOiBcImRlcGVuZGVuY2llc1wiLFxuICAgICAgfSxcbiAgICBdKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb3BpZXMgdGhlIGdpdmVuIHN0ZXAgaW50byB0aGUgZ2l2ZW4gdGFzay4gU3RlcCBhbmQgVGFzayBtdXN0IGJlIGZyb20gdGhlIGdpdmVuIFByb2plY3RcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByaXZhdGUgY29weVN0ZXBJbnRvVGFzayhzdGVwOiBUYXNrU3RlcCwgdGFzazogVGFzaywgcHJvamVjdDogUHJvamVjdCkge1xuICAgIGlmIChzdGVwLmV4ZWMpIHtcbiAgICAgIHRhc2suZXhlYyhzdGVwLmV4ZWMsIHsgbmFtZTogc3RlcC5uYW1lLCBjd2Q6IHN0ZXAuY3dkIH0pO1xuICAgIH0gZWxzZSBpZiAoc3RlcC5zYXkpIHtcbiAgICAgIHRhc2suc2F5KHN0ZXAuc2F5LCB7IG5hbWU6IHN0ZXAubmFtZSwgY3dkOiBzdGVwLmN3ZCB9KTtcbiAgICB9IGVsc2UgaWYgKHN0ZXAuc3Bhd24pIHtcbiAgICAgIGNvbnN0IHN0ZXBUb1NwYXduID0gcHJvamVjdC50YXNrcy50cnlGaW5kKHN0ZXAuc3Bhd24pO1xuICAgICAgaWYgKHN0ZXBUb1NwYXduKSB7XG4gICAgICAgIHRhc2suc3Bhd24oc3RlcFRvU3Bhd24sIHsgbmFtZTogc3RlcC5uYW1lLCBjd2Q6IHN0ZXAuY3dkIH0pO1xuICAgICAgfVxuICAgIH0gZWxzZSBpZiAoc3RlcC5idWlsdGluKSB7XG4gICAgICB0YXNrLmJ1aWx0aW4oc3RlcC5idWlsdGluKTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBEZXRlcm1pbmVzIGlmIHRoZSBwYXNzZWQgaW4gcHJvamVjdCBpcyBvZiB0eXBlIE5vZGVQcm9qZWN0LlxuICpcbiAqIEBwYXJhbSBwcm9qZWN0IFByb2plY3QgaW5zdGFuY2UuXG4gKiBAcmV0dXJucyB0cnVlIGlmIHRoZSBwcm9qZWN0IGluc3RhbmNlIGlzIG9mIHR5cGUgTm9kZVByb2plY3QuXG4gKi9cbmZ1bmN0aW9uIGlzTm9kZVByb2plY3QocHJvamVjdDogYW55KSB7XG4gIHJldHVybiBwcm9qZWN0IGluc3RhbmNlb2YgTm9kZVByb2plY3QgfHwgcHJvamVjdC5wYWNrYWdlO1xufVxuXG5mdW5jdGlvbiBnZXRQbHVnaW5QYXRoKCkge1xuICByZXR1cm4gcGF0aC5qb2luKF9fZGlybmFtZSwgXCJwbHVnaW4vbngtbW9ub3JlcG8tcGx1Z2luLmpzXCIpO1xufVxuIl19