"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./cross-region-support-stack");
const full_action_descriptor_1 = require("./full-action-descriptor");
const stage_1 = require("./stage");
const validation_1 = require("./validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * const pipeline = new Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ stageName: 'Source' });
 *
 * // add a source action to the stage
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   outputArtifactName: 'SourceArtifact',
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            const encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                // remove the key - there is a grace period of a few days before it's gone for good,
                // that should be enough for any emergency access to the bucket artifacts
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: s3.BucketEncryption.KMS,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
            // add an alias to make finding the key in the console easier
            new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                aliasName: this.generateNameForDefaultBucketKeyAlias(),
                targetKey: encryptionKey,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.anyValue({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).parseArn(pipelineArn).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Access one of the pipeline's stages by stage name
     */
    stage(stageName) {
        for (const stage of this._stages) {
            if (stage.stageName === stageName) {
                return stage;
            }
        }
        throw new Error(`Pipeline does not contain a stage named '${stageName}'. Available stages: ${this._stages.map(s => s.stageName).join(', ')}`);
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @experimental
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(action);
        // get the role for the given action
        const actionRole = this.getRoleForAction(stage, action, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(action.actionProperties.variablesNamespace);
        // bind the Action
        const actionConfig = action.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRegion;
        let otherStack;
        const actionResource = action.actionProperties.resource;
        if (actionResource) {
            const pipelineAndActionRegionComparison = core_1.Token.compareStrings(this.env.region, actionResource.env.region);
            const pipelineAndActionInDifferentRegions = pipelineAndActionRegionComparison === core_1.TokenComparison.ONE_UNRESOLVED ||
                pipelineAndActionRegionComparison === core_1.TokenComparison.DIFFERENT;
            if (pipelineAndActionInDifferentRegions) {
                actionRegion = actionResource.env.region;
                // if the resource is from a different stack in another region but the same account,
                // use that stack as home for the cross-region support resources
                const actionResourceStack = core_1.Stack.of(actionResource);
                const actionResourceAndItsStackRegionComparison = core_1.Token.compareStrings(actionResource.env.region, actionResourceStack.region);
                const actionResourceInSameRegionAsItsStack = actionResourceAndItsStackRegionComparison === core_1.TokenComparison.SAME ||
                    actionResourceAndItsStackRegionComparison === core_1.TokenComparison.BOTH_UNRESOLVED;
                const pipelineAndActionResourceStackAccountComparison = core_1.Token.compareStrings(this.env.account, actionResourceStack.account);
                const pipelineAndActionResourceStackInSameAccount = pipelineAndActionResourceStackAccountComparison === core_1.TokenComparison.SAME ||
                    pipelineAndActionResourceStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED;
                if (pipelineAndActionResourceStackInSameAccount && actionResourceInSameRegionAsItsStack) {
                    otherStack = actionResourceStack;
                }
            }
        }
        else {
            actionRegion = action.actionProperties.region;
        }
        // if actionRegion is undefined,
        // it means the action is in the same region as the pipeline -
        // so, just return the artifactBucket
        if (!actionRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // get the region the Pipeline itself is in
        const pipelineRegion = this.requireRegion();
        // if the action is in the same region as the pipeline, nothing to do
        if (actionRegion === pipelineRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        // the stack containing the replication bucket must be deployed before the pipeline
        pipelineStack.addDependency(crossRegionSupport.stack);
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: actionRegion,
        };
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id);
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.requireApp();
        const supportStackId = `cross-region-stack-${pipelineAccount}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = this.node.uniqueId;
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        if (!this.artifactBucket.encryptionKey) {
            throw new Error('The Pipeline is being used in a cross-account manner, ' +
                'but its artifact bucket does not have a KMS key defined. ' +
                'A KMS key is required for a cross-account Pipeline. ' +
                'Make sure to pass a Bucket with a Key when creating the Pipeline');
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${this.node.uniqueId}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const pipelineStack = core_1.Stack.of(this);
        if (action.actionProperties.resource) {
            const resourceStack = core_1.Stack.of(action.actionProperties.resource);
            // check if resource is from a different account
            if (pipelineStack.account === resourceStack.account) {
                return undefined;
            }
            else {
                this._crossAccountSupport[resourceStack.account] = resourceStack;
                return resourceStack;
            }
        }
        if (!action.actionProperties.account) {
            return undefined;
        }
        const targetAccount = action.actionProperties.account;
        // check whether the account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
        }
        // check whether the pipeline account is a static string
        if (core_1.Token.isUnresolved(pipelineStack.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        if (pipelineStack.account === targetAccount) {
            return undefined;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.requireApp();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: action.actionProperties.region ? action.actionProperties.region : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
}
exports.Pipeline = Pipeline;
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
//# sourceMappingURL=data:application/json;base64,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