"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const lambda_helper_1 = require("./lambda-helper");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    const cfnCfDistribution = cfDistribution.node.defaultChild;
    cfnCfDistribution.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W70',
                    reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
                }]
        }
    };
    return cfDistribution;
}
// Lambda@Edge function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultLambdaEdgeFunction(scope) {
    const edgeLambdaFunc = lambda_helper_1.deployLambdaFunction(scope, {
        code: new lambda.InlineCode("exports.handler = (event, context, callback) => { \
          const response = event.Records[0].cf.response; \
          const headers = response.headers; \
          headers['x-xss-protection'] = [ \
            { \
              key: 'X-XSS-Protection', \
              value: '1; mode=block' \
            } \
          ]; \
          headers['x-frame-options'] = [ \
            { \
              key: 'X-Frame-Options', \
              value: 'DENY' \
            } \
          ]; \
          headers['x-content-type-options'] = [ \
            { \
              key: 'X-Content-Type-Options', \
              value: 'nosniff' \
            } \
          ]; \
          headers['strict-transport-security'] = [ \
            { \
              key: 'Strict-Transport-Security', \
              value: 'max-age=63072000; includeSubdomains; preload' \
            } \
          ]; \
          headers['referrer-policy'] = [ \
            { \
              key: 'Referrer-Policy', \
              value: 'same-origin' \
            } \
          ]; \
          headers['content-security-policy'] = [ \
            { \
              key: 'Content-Security-Policy', \
              value: \"default-src 'none'; base-uri 'self'; img-src 'self'; script-src 'self'; style-src 'self' https:; object-src 'none'; frame-ancestors 'none'; font-src 'self' https:; form-action 'self'; manifest-src 'self'; connect-src 'self'\" \
             } \
          ]; \
          callback(null, response); \
        };"),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    }, 'SetHttpSecurityHeaders');
    // Switching from cloudfront.CloudFrontWebDistribution -> cloudfront.Distribution breaks the Lamba@Edge as it does not automatically update
    // the lambda role AssumePolicy for 'edgelambda.amazonaws.com'
    if (edgeLambdaFunc.role && edgeLambdaFunc.role instanceof iam.Role && edgeLambdaFunc.role.assumeRolePolicy) {
        edgeLambdaFunc.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
        }));
    }
    return edgeLambdaFunc;
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders) {
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    let defaultprops;
    let edgeLambdaVersion;
    let loggingBucket;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.enableLogging && cloudFrontDistributionProps.logBucket) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, cloudFrontDistributionProps.logBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, true) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, edgeLambdaVersion, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders) {
    let defaultprops;
    let edgeLambdaVersion;
    let loggingBucket;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.enableLogging && cloudFrontDistributionProps.logBucket) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, cloudFrontDistributionProps.logBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, true) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    const sourceBucketPolicy = bucketPolicy.node.findChild('Resource');
    sourceBucketPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'F16',
                    reason: `Public website bucket policy requires a wildcard principal`
                }]
        }
    };
    return [cfDistribution, edgeLambdaVersion, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders) {
    let defaultprops;
    let originRequestPolicy;
    let loggingBucket;
    let edgeLambdaVersion;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, 'SetHttpSecurityHeadersVersion', {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.enableLogging && cloudFrontDistributionProps.logBucket) {
        loggingBucket = cloudFrontDistributionProps.logBucket;
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
    }
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, _httpSecurityHeaders, cloudFrontDistributionProps === null || cloudFrontDistributionProps === void 0 ? void 0 : cloudFrontDistributionProps.customHeaders, edgeLambdaVersion);
    let cfprops;
    if (cloudFrontDistributionProps) {
        cfprops = utils_1.overrideProps(defaultprops, cloudFrontDistributionProps, true);
    }
    else {
        cfprops = defaultprops;
    }
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, loggingBucket, originRequestPolicy, edgeLambdaVersion];
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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