"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupCognitoForElasticSearch = exports.buildIdentityPool = exports.buildUserPoolClient = exports.buildUserPool = void 0;
const cognito = require("@aws-cdk/aws-cognito");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cognito_defaults_1 = require("./cognito-defaults");
function buildUserPool(scope, userPoolProps) {
    let cognitoUserPoolProps;
    if (userPoolProps) {
        cognitoUserPoolProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolProps, userPoolProps);
    }
    else {
        cognitoUserPoolProps = cognito_defaults_1.DefaultUserPoolProps;
    }
    const userPool = new cognito.UserPool(scope, 'CognitoUserPool', cognitoUserPoolProps);
    // Set the advancedSecurityMode to ENFORCED
    const cfnUserPool = userPool.node.findChild('Resource');
    cfnUserPool.userPoolAddOns = {
        advancedSecurityMode: 'ENFORCED'
    };
    // Add Cfn Nag suppress for the cognito SMS role policy
    const userPoolSmsRole = userPool.node.tryFindChild('smsRole');
    if (userPoolSmsRole) {
        const cfnuserPoolSmsRole = userPoolSmsRole.node.defaultChild;
        cfnuserPoolSmsRole.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W11',
                        reason: `Allowing * resource on permissions policy since its used by Cognito to send SMS messages via sns:Publish`
                    }]
            }
        };
    }
    return userPool;
}
exports.buildUserPool = buildUserPool;
function buildUserPoolClient(scope, userPool, cognitoUserPoolClientProps) {
    let userPoolClientProps;
    if (cognitoUserPoolClientProps) {
        userPoolClientProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolClientProps(userPool), cognitoUserPoolClientProps);
    }
    else {
        userPoolClientProps = cognito_defaults_1.DefaultUserPoolClientProps(userPool);
    }
    return new cognito.UserPoolClient(scope, 'CognitoUserPoolClient', userPoolClientProps);
}
exports.buildUserPoolClient = buildUserPoolClient;
function buildIdentityPool(scope, userpool, userpoolclient, identityPoolProps) {
    let cognitoIdentityPoolProps = cognito_defaults_1.DefaultIdentityPoolProps(userpoolclient.userPoolClientId, userpool.userPoolProviderName);
    if (identityPoolProps) {
        cognitoIdentityPoolProps = utils_1.overrideProps(cognitoIdentityPoolProps, identityPoolProps);
    }
    const idPool = new cognito.CfnIdentityPool(scope, 'CognitoIdentityPool', cognitoIdentityPoolProps);
    return idPool;
}
exports.buildIdentityPool = buildIdentityPool;
function setupCognitoForElasticSearch(scope, domainName, options) {
    // Create the domain for Cognito UserPool
    const userpooldomain = new cognito.CfnUserPoolDomain(scope, 'UserPoolDomain', {
        domain: domainName,
        userPoolId: options.userpool.userPoolId
    });
    userpooldomain.addDependsOn(options.userpool.node.findChild('Resource'));
    // Setup the IAM Role for Cognito Authorized Users
    const cognitoPrincipal = new iam.FederatedPrincipal('cognito-identity.amazonaws.com', {
        'StringEquals': { 'cognito-identity.amazonaws.com:aud': options.identitypool.ref },
        'ForAnyValue:StringLike': { 'cognito-identity.amazonaws.com:amr': 'authenticated' }
    }, 'sts:AssumeRoleWithWebIdentity');
    const cognitoAuthorizedRole = new iam.Role(scope, 'CognitoAuthorizedRole', {
        assumedBy: cognitoPrincipal,
        inlinePolicies: {
            CognitoAccessPolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'es:ESHttp*'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`]
                    })
                ]
            })
        }
    });
    // Attach the IAM Role for Cognito Authorized Users
    new cognito.CfnIdentityPoolRoleAttachment(scope, 'IdentityPoolRoleMapping', {
        identityPoolId: options.identitypool.ref,
        roles: {
            authenticated: cognitoAuthorizedRole.roleArn
        }
    });
    return cognitoAuthorizedRole;
}
exports.setupCognitoForElasticSearch = setupCognitoForElasticSearch;
//# sourceMappingURL=data:application/json;base64,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