"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        });
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_10_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_12_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    cfnLambdafunction.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
                },
                {
                    id: 'W89',
                    reason: `This is not a rule for the general case, just for specific use cases/industries`
                },
                {
                    id: 'W92',
                    reason: `Impossible for us to define the correct concurrency for clients`
                }]
        }
    };
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        cfnLambdafunctionDefPolicy.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
                    }]
            }
        };
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
//# sourceMappingURL=data:application/json;base64,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