"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
// Imports
const sns = require("@aws-cdk/aws-sns");
const kms = require("@aws-cdk/aws-kms");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": core_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
function buildTopic(scope, props) {
    if (!props.existingTopicObj) {
        // Setup the topic properties
        let snsTopicProps;
        if (props.topicProps) {
            // If property overrides have been provided, incorporate them and deploy
            snsTopicProps = utils_1.overrideProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
        }
        else {
            // If no property overrides, deploy using the default configuration
            snsTopicProps = sns_defaults_1.DefaultSnsTopicProps;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey === undefined || props.enableEncryptionWithCustomerManagedKey === false) {
            // Retrieve SNS managed key to encrypt the SNS Topic
            const awsManagedKey = kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns');
            snsTopicProps.masterKey = awsManagedKey;
        }
        else {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                snsTopicProps.masterKey = props.encryptionKey;
            }
            else {
                // Create a new Customer Managed KMS key
                snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        // Create the SNS Topic
        const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
        applySecureTopicPolicy(topic);
        return [topic, snsTopicProps.masterKey];
    }
    else {
        return [props.existingTopicObj];
    }
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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