"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    var _a;
    if (props === null || props === void 0 ? void 0 : props.existingVpc) {
        return props === null || props === void 0 ? void 0 : props.existingVpc;
    }
    let cumulativeProps = props === null || props === void 0 ? void 0 : props.defaultVpcProps;
    if (props === null || props === void 0 ? void 0 : props.userVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.userVpcProps);
    }
    if (props === null || props === void 0 ? void 0 : props.constructVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.constructVpcProps);
    }
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        cfnSubnet.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W33',
                        reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
                    }]
            }
        };
    });
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = (_a = flowLog.logGroup) === null || _a === void 0 ? void 0 : _a.node.defaultChild;
    cfnLogGroup.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W84',
                    reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
                }]
        }
    };
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEPFUNCTIONS"] = "STEPFUNCTIONS";
    ServiceEndpointTypes["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypes["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypes["SSM"] = "SSM";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: ServiceEndpointTypes.SAGEMAKER_RUNTIME,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: ServiceEndpointTypes.SECRETS_MANAGER,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: ServiceEndpointTypes.SSM,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SSM,
    },
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (!vpc.node.children.some((child) => child.node.id === interfaceTag)) {
        const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
        if (!service) {
            throw new Error("Unsupported Service sent to AddServiceEndpoint");
        }
        if (service.endpointType === EndpointTypes.GATEWAY) {
            vpc.addGatewayEndpoint(interfaceTag, {
                service: service.endpointGatewayService,
            });
        }
        if (service.endpointType === EndpointTypes.INTERFACE) {
            const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceEndpointDefaultSecurityGroup", {
                vpc,
                allowAllOutbound: true,
            }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
            vpc.addInterfaceEndpoint(interfaceTag, {
                service: service.endpointInterfaceService,
                securityGroups: [endpointDefaultSecurityGroup],
            });
        }
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBjLWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInZwYy1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOzs7QUFFSCx3Q0FBd0M7QUFHeEMsbUVBQTZEO0FBQzdELG1DQUF3QztBQXNCeEMsU0FBZ0IsUUFBUSxDQUFDLEtBQWdCLEVBQUUsS0FBb0I7O0lBQzdELElBQUksS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFdBQVcsRUFBRTtRQUN0QixPQUFPLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxXQUFXLENBQUM7S0FDM0I7SUFFRCxJQUFJLGVBQWUsR0FBaUIsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLGVBQWUsQ0FBQztJQUUzRCxJQUFJLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxZQUFZLEVBQUU7UUFDdkIsZUFBZSxHQUFHLHFCQUFhLENBQUMsZUFBZSxFQUFFLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxZQUFZLENBQUMsQ0FBQztLQUN2RTtJQUVELElBQUksS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLGlCQUFpQixFQUFFO1FBQzVCLGVBQWUsR0FBRyxxQkFBYSxDQUM3QixlQUFlLEVBQ2YsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLGlCQUFpQixDQUN6QixDQUFDO0tBQ0g7SUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztJQUV2RCxnR0FBZ0c7SUFDaEcsTUFBTSxPQUFPLEdBQWdCLEdBQUcsQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUM7SUFFdkQsOEhBQThIO0lBQzlILEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7UUFDbkMsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUE2QixDQUFDO1FBQzVELFNBQVMsQ0FBQyxVQUFVLENBQUMsUUFBUSxHQUFHO1lBQzlCLE9BQU8sRUFBRTtnQkFDUCxpQkFBaUIsRUFBRSxDQUFDO3dCQUNsQixFQUFFLEVBQUUsS0FBSzt3QkFDVCxNQUFNLEVBQUUsOERBQThEO3FCQUN2RSxDQUFDO2FBQ0g7U0FDRixDQUFDO0lBQ0osQ0FBQyxDQUFDLENBQUM7SUFFSCx5RUFBeUU7SUFDekUsTUFBTSxXQUFXLEdBQWdCLE1BQUEsT0FBTyxDQUFDLFFBQVEsMENBQUUsSUFBSSxDQUFDLFlBQTJCLENBQUM7SUFFcEYsV0FBVyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEdBQUc7UUFDaEMsT0FBTyxFQUFFO1lBQ1AsaUJBQWlCLEVBQUUsQ0FBQztvQkFDbEIsRUFBRSxFQUFFLEtBQUs7b0JBQ1QsTUFBTSxFQUFFLDJIQUEySDtpQkFDcEksQ0FBQztTQUNIO0tBQ0YsQ0FBQztJQUVGLE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQWpERCw0QkFpREM7QUFFRCxJQUFZLG9CQVNYO0FBVEQsV0FBWSxvQkFBb0I7SUFDOUIsd0NBQWdCLENBQUE7SUFDaEIsbUNBQVcsQ0FBQTtJQUNYLG1DQUFXLENBQUE7SUFDWCxpQ0FBUyxDQUFBO0lBQ1QsdURBQStCLENBQUE7SUFDL0IsK0RBQXVDLENBQUE7SUFDdkMsMkRBQW1DLENBQUE7SUFDbkMsbUNBQVcsQ0FBQTtBQUNiLENBQUMsRUFUVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQVMvQjtBQUVELElBQUssYUFHSjtBQUhELFdBQUssYUFBYTtJQUNoQixvQ0FBbUIsQ0FBQTtJQUNuQix3Q0FBdUIsQ0FBQTtBQUN6QixDQUFDLEVBSEksYUFBYSxLQUFiLGFBQWEsUUFHakI7QUFTRCxNQUFNLGdCQUFnQixHQUF5QjtJQUM3QztRQUNFLFlBQVksRUFBRSxvQkFBb0IsQ0FBQyxRQUFRO1FBQzNDLFlBQVksRUFBRSxhQUFhLENBQUMsT0FBTztRQUNuQyxzQkFBc0IsRUFBRSxHQUFHLENBQUMsNEJBQTRCLENBQUMsUUFBUTtLQUNsRTtJQUNEO1FBQ0UsWUFBWSxFQUFFLG9CQUFvQixDQUFDLEVBQUU7UUFDckMsWUFBWSxFQUFFLGFBQWEsQ0FBQyxPQUFPO1FBQ25DLHNCQUFzQixFQUFFLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxFQUFFO0tBQzVEO0lBQ0Q7UUFDRSxZQUFZLEVBQUUsb0JBQW9CLENBQUMsR0FBRztRQUN0QyxZQUFZLEVBQUUsYUFBYSxDQUFDLFNBQVM7UUFDckMsd0JBQXdCLEVBQUUsR0FBRyxDQUFDLDhCQUE4QixDQUFDLEdBQUc7S0FDakU7SUFDRDtRQUNFLFlBQVksRUFBRSxvQkFBb0IsQ0FBQyxHQUFHO1FBQ3RDLFlBQVksRUFBRSxhQUFhLENBQUMsU0FBUztRQUNyQyx3QkFBd0IsRUFBRSxHQUFHLENBQUMsOEJBQThCLENBQUMsR0FBRztLQUNqRTtJQUNEO1FBQ0UsWUFBWSxFQUFFLG9CQUFvQixDQUFDLGlCQUFpQjtRQUNwRCxZQUFZLEVBQUUsYUFBYSxDQUFDLFNBQVM7UUFDckMsd0JBQXdCLEVBQUUsR0FBRyxDQUFDLDhCQUE4QixDQUFDLGlCQUFpQjtLQUMvRTtJQUNEO1FBQ0UsWUFBWSxFQUFFLG9CQUFvQixDQUFDLGVBQWU7UUFDbEQsWUFBWSxFQUFFLGFBQWEsQ0FBQyxTQUFTO1FBQ3JDLHdCQUF3QixFQUFFLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxlQUFlO0tBQzdFO0lBQ0Q7UUFDRSxZQUFZLEVBQUUsb0JBQW9CLENBQUMsR0FBRztRQUN0QyxZQUFZLEVBQUUsYUFBYSxDQUFDLFNBQVM7UUFDckMsd0JBQXdCLEVBQUUsR0FBRyxDQUFDLDhCQUE4QixDQUFDLEdBQUc7S0FDakU7Q0FDRixDQUFDO0FBRUYsU0FBZ0IscUJBQXFCLENBQ25DLEtBQWdCLEVBQ2hCLEdBQWEsRUFDYixZQUFrQztJQUVsQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxZQUFZLENBQUMsRUFBRTtRQUN0RSxNQUFNLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQyxJQUFJLENBQ25DLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxRQUFRLENBQUMsWUFBWSxLQUFLLFlBQVksQ0FDckQsQ0FBQztRQUVGLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDWixNQUFNLElBQUksS0FBSyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7U0FDbkU7UUFFRCxJQUFJLE9BQU8sQ0FBQyxZQUFZLEtBQUssYUFBYSxDQUFDLE9BQU8sRUFBRTtZQUNsRCxHQUFHLENBQUMsa0JBQWtCLENBQUMsWUFBWSxFQUFFO2dCQUNuQyxPQUFPLEVBQUUsT0FBTyxDQUFDLHNCQUEwRDthQUM1RSxDQUFDLENBQUM7U0FDSjtRQUNELElBQUksT0FBTyxDQUFDLFlBQVksS0FBSyxhQUFhLENBQUMsU0FBUyxFQUFFO1lBRXBELE1BQU0sNEJBQTRCLEdBQUcsMENBQWtCLENBQ3JELEtBQUssRUFDTCxxQ0FBcUMsRUFDckM7Z0JBQ0UsR0FBRztnQkFDSCxnQkFBZ0IsRUFBRSxJQUFJO2FBQ3ZCLEVBQ0QsQ0FBQyxFQUFFLElBQUksRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFDMUUsRUFBRSxDQUNILENBQUM7WUFFRixHQUFHLENBQUMsb0JBQW9CLENBQUMsWUFBWSxFQUFFO2dCQUNyQyxPQUFPLEVBQUUsT0FBTyxDQUFDLHdCQUE4RDtnQkFDL0UsY0FBYyxFQUFFLENBQUUsNEJBQTRCLENBQUU7YUFDakQsQ0FBQyxDQUFDO1NBQ0o7S0FDRjtJQUVELE9BQU87QUFDVCxDQUFDO0FBeENELHNEQXdDQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIxIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgZWMyIGZyb20gXCJAYXdzLWNkay9hd3MtZWMyXCI7XG5pbXBvcnQgeyBDZm5Mb2dHcm91cCB9IGZyb20gXCJAYXdzLWNkay9hd3MtbG9nc1wiO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcIkBhd3MtY2RrL2NvcmVcIjtcbmltcG9ydCB7IGJ1aWxkU2VjdXJpdHlHcm91cCB9IGZyb20gXCIuL3NlY3VyaXR5LWdyb3VwLWhlbHBlclwiO1xuaW1wb3J0IHsgb3ZlcnJpZGVQcm9wcyB9IGZyb20gXCIuL3V0aWxzXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVpbGRWcGNQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXhpc3RpbmdWcGM/OiBlYzIuSVZwYztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlZmF1bHRWcGNQcm9wczogZWMyLlZwY1Byb3BzO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyVnBjUHJvcHM/OiBlYzIuVnBjUHJvcHM7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbnN0cnVjdFZwY1Byb3BzPzogZWMyLlZwY1Byb3BzO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRWcGMoc2NvcGU6IENvbnN0cnVjdCwgcHJvcHM6IEJ1aWxkVnBjUHJvcHMpOiBlYzIuSVZwYyB7XG4gIGlmIChwcm9wcz8uZXhpc3RpbmdWcGMpIHtcbiAgICByZXR1cm4gcHJvcHM/LmV4aXN0aW5nVnBjO1xuICB9XG5cbiAgbGV0IGN1bXVsYXRpdmVQcm9wczogZWMyLlZwY1Byb3BzID0gcHJvcHM/LmRlZmF1bHRWcGNQcm9wcztcblxuICBpZiAocHJvcHM/LnVzZXJWcGNQcm9wcykge1xuICAgIGN1bXVsYXRpdmVQcm9wcyA9IG92ZXJyaWRlUHJvcHMoY3VtdWxhdGl2ZVByb3BzLCBwcm9wcz8udXNlclZwY1Byb3BzKTtcbiAgfVxuXG4gIGlmIChwcm9wcz8uY29uc3RydWN0VnBjUHJvcHMpIHtcbiAgICBjdW11bGF0aXZlUHJvcHMgPSBvdmVycmlkZVByb3BzKFxuICAgICAgY3VtdWxhdGl2ZVByb3BzLFxuICAgICAgcHJvcHM/LmNvbnN0cnVjdFZwY1Byb3BzXG4gICAgKTtcbiAgfVxuXG4gIGNvbnN0IHZwYyA9IG5ldyBlYzIuVnBjKHNjb3BlLCBcIlZwY1wiLCBjdW11bGF0aXZlUHJvcHMpO1xuXG4gIC8vIEFkZCBWUEMgRmxvd0xvZ3Mgd2l0aCB0aGUgZGVmYXVsdCBzZXR0aW5nIG9mIHRyYWZmaWNUeXBlOkFMTCBhbmQgZGVzdGluYXRpb246IENsb3VkV2F0Y2ggTG9nc1xuICBjb25zdCBmbG93TG9nOiBlYzIuRmxvd0xvZyA9IHZwYy5hZGRGbG93TG9nKFwiRmxvd0xvZ1wiKTtcblxuICAvLyBBZGQgQ2ZuIE5hZyBzdXBwcmVzc2lvbiBmb3IgUFVCTElDIHN1Ym5ldHMgdG8gc3VwcHJlc3MgV0FSTiBXMzM6IEVDMiBTdWJuZXQgc2hvdWxkIG5vdCBoYXZlIE1hcFB1YmxpY0lwT25MYXVuY2ggc2V0IHRvIHRydWVcbiAgdnBjLnB1YmxpY1N1Ym5ldHMuZm9yRWFjaCgoc3VibmV0KSA9PiB7XG4gICAgY29uc3QgY2ZuU3VibmV0ID0gc3VibmV0Lm5vZGUuZGVmYXVsdENoaWxkIGFzIGVjMi5DZm5TdWJuZXQ7XG4gICAgY2ZuU3VibmV0LmNmbk9wdGlvbnMubWV0YWRhdGEgPSB7XG4gICAgICBjZm5fbmFnOiB7XG4gICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbe1xuICAgICAgICAgIGlkOiAnVzMzJyxcbiAgICAgICAgICByZWFzb246ICdBbGxvdyBQdWJsaWMgU3VibmV0cyB0byBoYXZlIE1hcFB1YmxpY0lwT25MYXVuY2ggc2V0IHRvIHRydWUnXG4gICAgICAgIH1dXG4gICAgICB9XG4gICAgfTtcbiAgfSk7XG5cbiAgLy8gQWRkIENmbiBOYWcgc3VwcHJlc3Npb24gZm9yIENsb3VkV2F0Y2hMb2dzIExvZ0dyb3VwcyBkYXRhIGlzIGVuY3J5cHRlZFxuICBjb25zdCBjZm5Mb2dHcm91cDogQ2ZuTG9nR3JvdXAgPSBmbG93TG9nLmxvZ0dyb3VwPy5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5Mb2dHcm91cDtcblxuICBjZm5Mb2dHcm91cC5jZm5PcHRpb25zLm1ldGFkYXRhID0ge1xuICAgIGNmbl9uYWc6IHtcbiAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbe1xuICAgICAgICBpZDogJ1c4NCcsXG4gICAgICAgIHJlYXNvbjogJ0J5IGRlZmF1bHQgQ2xvdWRXYXRjaExvZ3MgTG9nR3JvdXBzIGRhdGEgaXMgZW5jcnlwdGVkIHVzaW5nIHRoZSBDbG91ZFdhdGNoIHNlcnZlci1zaWRlIGVuY3J5cHRpb24ga2V5cyAoQVdTIE1hbmFnZWQgS2V5cyknXG4gICAgICB9XVxuICAgIH1cbiAgfTtcblxuICByZXR1cm4gdnBjO1xufVxuXG5leHBvcnQgZW51bSBTZXJ2aWNlRW5kcG9pbnRUeXBlcyB7XG4gIERZTkFNT0RCID0gXCJEREJcIixcbiAgU05TID0gXCJTTlNcIixcbiAgU1FTID0gXCJTUVNcIixcbiAgUzMgPSBcIlMzXCIsXG4gIFNURVBGVU5DVElPTlMgPSBcIlNURVBGVU5DVElPTlNcIixcbiAgU0FHRU1BS0VSX1JVTlRJTUUgPSBcIlNBR0VNQUtFUl9SVU5USU1FXCIsXG4gIFNFQ1JFVFNfTUFOQUdFUiA9IFwiU0VDUkVUU19NQU5BR0VSXCIsXG4gIFNTTSA9IFwiU1NNXCIsXG59XG5cbmVudW0gRW5kcG9pbnRUeXBlcyB7XG4gIEdBVEVXQVkgPSBcIkdhdGV3YXlcIixcbiAgSU5URVJGQUNFID0gXCJJbnRlcmZhY2VcIixcbn1cblxuaW50ZXJmYWNlIEVuZHBvaW50RGVmaW5pdGlvbiB7XG4gIGVuZHBvaW50TmFtZTogU2VydmljZUVuZHBvaW50VHlwZXM7XG4gIGVuZHBvaW50VHlwZTogRW5kcG9pbnRUeXBlcztcbiAgZW5kcG9pbnRHYXRld2F5U2VydmljZT86IGVjMi5HYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlO1xuICBlbmRwb2ludEludGVyZmFjZVNlcnZpY2U/OiBlYzIuSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlO1xufVxuXG5jb25zdCBlbmRwb2ludFNldHRpbmdzOiBFbmRwb2ludERlZmluaXRpb25bXSA9IFtcbiAge1xuICAgIGVuZHBvaW50TmFtZTogU2VydmljZUVuZHBvaW50VHlwZXMuRFlOQU1PREIsXG4gICAgZW5kcG9pbnRUeXBlOiBFbmRwb2ludFR5cGVzLkdBVEVXQVksXG4gICAgZW5kcG9pbnRHYXRld2F5U2VydmljZTogZWMyLkdhdGV3YXlWcGNFbmRwb2ludEF3c1NlcnZpY2UuRFlOQU1PREIsXG4gIH0sXG4gIHtcbiAgICBlbmRwb2ludE5hbWU6IFNlcnZpY2VFbmRwb2ludFR5cGVzLlMzLFxuICAgIGVuZHBvaW50VHlwZTogRW5kcG9pbnRUeXBlcy5HQVRFV0FZLFxuICAgIGVuZHBvaW50R2F0ZXdheVNlcnZpY2U6IGVjMi5HYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlMzLFxuICB9LFxuICB7XG4gICAgZW5kcG9pbnROYW1lOiBTZXJ2aWNlRW5kcG9pbnRUeXBlcy5TTlMsXG4gICAgZW5kcG9pbnRUeXBlOiBFbmRwb2ludFR5cGVzLklOVEVSRkFDRSxcbiAgICBlbmRwb2ludEludGVyZmFjZVNlcnZpY2U6IGVjMi5JbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UuU05TLFxuICB9LFxuICB7XG4gICAgZW5kcG9pbnROYW1lOiBTZXJ2aWNlRW5kcG9pbnRUeXBlcy5TUVMsXG4gICAgZW5kcG9pbnRUeXBlOiBFbmRwb2ludFR5cGVzLklOVEVSRkFDRSxcbiAgICBlbmRwb2ludEludGVyZmFjZVNlcnZpY2U6IGVjMi5JbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UuU1FTLFxuICB9LFxuICB7XG4gICAgZW5kcG9pbnROYW1lOiBTZXJ2aWNlRW5kcG9pbnRUeXBlcy5TQUdFTUFLRVJfUlVOVElNRSxcbiAgICBlbmRwb2ludFR5cGU6IEVuZHBvaW50VHlwZXMuSU5URVJGQUNFLFxuICAgIGVuZHBvaW50SW50ZXJmYWNlU2VydmljZTogZWMyLkludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZS5TQUdFTUFLRVJfUlVOVElNRSxcbiAgfSxcbiAge1xuICAgIGVuZHBvaW50TmFtZTogU2VydmljZUVuZHBvaW50VHlwZXMuU0VDUkVUU19NQU5BR0VSLFxuICAgIGVuZHBvaW50VHlwZTogRW5kcG9pbnRUeXBlcy5JTlRFUkZBQ0UsXG4gICAgZW5kcG9pbnRJbnRlcmZhY2VTZXJ2aWNlOiBlYzIuSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlNFQ1JFVFNfTUFOQUdFUixcbiAgfSxcbiAge1xuICAgIGVuZHBvaW50TmFtZTogU2VydmljZUVuZHBvaW50VHlwZXMuU1NNLFxuICAgIGVuZHBvaW50VHlwZTogRW5kcG9pbnRUeXBlcy5JTlRFUkZBQ0UsXG4gICAgZW5kcG9pbnRJbnRlcmZhY2VTZXJ2aWNlOiBlYzIuSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlNTTSxcbiAgfSxcbl07XG5cbmV4cG9ydCBmdW5jdGlvbiBBZGRBd3NTZXJ2aWNlRW5kcG9pbnQoXG4gIHNjb3BlOiBDb25zdHJ1Y3QsXG4gIHZwYzogZWMyLklWcGMsXG4gIGludGVyZmFjZVRhZzogU2VydmljZUVuZHBvaW50VHlwZXNcbikge1xuICBpZiAoIXZwYy5ub2RlLmNoaWxkcmVuLnNvbWUoKGNoaWxkKSA9PiBjaGlsZC5ub2RlLmlkID09PSBpbnRlcmZhY2VUYWcpKSB7XG4gICAgY29uc3Qgc2VydmljZSA9IGVuZHBvaW50U2V0dGluZ3MuZmluZChcbiAgICAgIChlbmRwb2ludCkgPT4gZW5kcG9pbnQuZW5kcG9pbnROYW1lID09PSBpbnRlcmZhY2VUYWdcbiAgICApO1xuXG4gICAgaWYgKCFzZXJ2aWNlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJVbnN1cHBvcnRlZCBTZXJ2aWNlIHNlbnQgdG8gQWRkU2VydmljZUVuZHBvaW50XCIpO1xuICAgIH1cblxuICAgIGlmIChzZXJ2aWNlLmVuZHBvaW50VHlwZSA9PT0gRW5kcG9pbnRUeXBlcy5HQVRFV0FZKSB7XG4gICAgICB2cGMuYWRkR2F0ZXdheUVuZHBvaW50KGludGVyZmFjZVRhZywge1xuICAgICAgICBzZXJ2aWNlOiBzZXJ2aWNlLmVuZHBvaW50R2F0ZXdheVNlcnZpY2UgYXMgZWMyLkdhdGV3YXlWcGNFbmRwb2ludEF3c1NlcnZpY2UsXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKHNlcnZpY2UuZW5kcG9pbnRUeXBlID09PSBFbmRwb2ludFR5cGVzLklOVEVSRkFDRSkge1xuXG4gICAgICBjb25zdCBlbmRwb2ludERlZmF1bHRTZWN1cml0eUdyb3VwID0gYnVpbGRTZWN1cml0eUdyb3VwKFxuICAgICAgICBzY29wZSxcbiAgICAgICAgXCJSZXBsYWNlRW5kcG9pbnREZWZhdWx0U2VjdXJpdHlHcm91cFwiLFxuICAgICAgICB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICAgIFt7IHBlZXI6IGVjMi5QZWVyLmlwdjQodnBjLnZwY0NpZHJCbG9jayksIGNvbm5lY3Rpb246IGVjMi5Qb3J0LnRjcCg0NDMpIH1dLFxuICAgICAgICBbXVxuICAgICAgKTtcblxuICAgICAgdnBjLmFkZEludGVyZmFjZUVuZHBvaW50KGludGVyZmFjZVRhZywge1xuICAgICAgICBzZXJ2aWNlOiBzZXJ2aWNlLmVuZHBvaW50SW50ZXJmYWNlU2VydmljZSBhcyBlYzIuSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlLFxuICAgICAgICBzZWN1cml0eUdyb3VwczogWyBlbmRwb2ludERlZmF1bHRTZWN1cml0eUdyb3VwIF0sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICByZXR1cm47XG59XG4iXX0=