"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const logs = require("@aws-cdk/aws-logs");
const cloudwatch_log_group_helper_1 = require("../lib/cloudwatch-log-group-helper");
const kms = require("@aws-cdk/aws-kms");
test('cw log group with default params', () => {
    const stack = new core_1.Stack();
    cloudwatch_log_group_helper_1.buildLogGroup(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('override cw log group props with encryptionKey and retention period', () => {
    const stack = new core_1.Stack();
    const key = new kms.Key(stack, 'mykey');
    cloudwatch_log_group_helper_1.buildLogGroup(stack, 'test-cw-logs-default', {
        encryptionKey: key,
        retention: logs.RetentionDays.FIVE_DAYS
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('override cw log group props with encryptionKey only', () => {
    const stack = new core_1.Stack();
    const key = new kms.Key(stack, 'mykey');
    cloudwatch_log_group_helper_1.buildLogGroup(stack, 'test-cw-logs-default', {
        encryptionKey: key
    });
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: 'W86',
                        reason: 'Retention period for CloudWatchLogs LogGroups are set to \'Never Expire\' to preserve customer data indefinitely'
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('override cw log group props with retention period only', () => {
    const stack = new core_1.Stack();
    cloudwatch_log_group_helper_1.buildLogGroup(stack, 'test-cw-logs-default', {
        retention: logs.RetentionDays.FIVE_DAYS
    });
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: 'W84',
                        reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
//# sourceMappingURL=data:application/json;base64,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