"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const defaults = require("../");
const core_1 = require("@aws-cdk/core");
const test_helper_1 = require("./test-helper");
const lambda = require("@aws-cdk/aws-lambda");
const sqs = require("@aws-cdk/aws-sqs");
const mediastore_defaults_1 = require("../lib/mediastore-defaults");
const mediastore = require("@aws-cdk/aws-mediastore");
const kinesis = require("@aws-cdk/aws-kinesis");
const sns = require("@aws-cdk/aws-sns");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const sagemaker_helper_1 = require("../lib/sagemaker-helper");
test('Test with valid props', () => {
    const props = {};
    defaults.CheckProps(props);
});
test('Test fail DynamoDB table check', () => {
    const stack = new core_1.Stack();
    const props = {
        existingTableObj: new dynamodb.Table(stack, 'placeholder', defaults.DefaultTableProps),
        dynamoTableProps: defaults.DefaultTableProps,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
});
test("Test fail Lambda function check", () => {
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
test("Test fail SQS Queue check", () => {
    const stack = new core_1.Stack();
    const props = {
        queueProps: {},
        existingQueueObj: new sqs.Queue(stack, 'placeholder', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide queueProps or existingQueueObj, but not both.\n');
});
test('Test fail Dead Letter Queue check', () => {
    const props = {
        deployDeadLetterQueue: false,
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If deployDeadLetterQueue is false then deadLetterQueueProps cannot be specified.\n');
});
test('Test fail Dead Letter Queue check with no deployDeadLetterQueue flag', () => {
    const props = {
        deadLetterQueueProps: {},
    };
    //  Should not flag an error
    defaults.CheckProps(props);
});
test("Test fail MediaStore container check", () => {
    const stack = new core_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, "placeholder", mediastore_defaults_1.MediaStoreContainerProps());
    const props = {
        mediaStoreContainerProps: mediastore_defaults_1.MediaStoreContainerProps(),
        existingMediaStoreContainerObj: mediaStoreContainer,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide mediaStoreContainerProps or existingMediaStoreContainerObj, but not both.\n");
});
test('Test fail Kinesis stream check', () => {
    const stack = new core_1.Stack();
    const stream = new kinesis.Stream(stack, 'placeholder', {});
    const props = {
        existingStreamObj: stream,
        kinesisStreamProps: {}
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
test('Test fail S3 check', () => {
    const stack = new core_1.Stack();
    const props = {
        existingBucketObj: test_helper_1.CreateScrapBucket(stack, {}),
        bucketProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
test('Test fail SNS topic check', () => {
    const stack = new core_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail SNS topic check with bad topic attribute name', () => {
    const stack = new core_1.Stack();
    const props = {
        topicsProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail Glue job check', () => {
    const stack = new core_1.Stack();
    const _jobRole = new iam.Role(stack, 'CustomETLJobRole', {
        assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
    });
    const jobProps = defaults.DefaultGlueJobProps(_jobRole, {
        name: 'placeholder',
        pythonVersion: '3',
        scriptLocation: 's3://fakelocation/script'
    }, 'testETLJob', {}, '1.0');
    const job = new glue.CfnJob(stack, 'placeholder', jobProps);
    const props = {
        glueJobProps: jobProps,
        existingGlueJob: job
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide glueJobProps or existingGlueJob, but not both.\n');
});
test('Test fail SageMaker endpoint check', () => {
    const stack = new core_1.Stack();
    // Build Sagemaker Inference Endpoint
    const modelProps = {
        primaryContainer: {
            image: "<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest",
            modelDataUrl: "s3://<bucket-name>/<prefix>/model.tar.gz",
        },
    };
    const [endpoint] = sagemaker_helper_1.BuildSagemakerEndpoint(stack, { modelProps });
    const props = {
        existingSagemakerEndpointObj: endpoint,
        endpointProps: {
            endpointConfigName: 'placeholder'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n');
});
test('Test fail Secret check', () => {
    const stack = new core_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
test('Test fail encryption key check', () => {
    const stack = new core_1.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        enableKeyRotation: false
    });
    const props = {
        encryptionKey: key,
        encryptionKeyProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n');
});
test('Test fail Vpc check with deployVpc', () => {
    const stack = new core_1.Stack();
    const props = {
        deployVpc: true,
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail Vpc check with vpcProps', () => {
    const stack = new core_1.Stack();
    const props = {
        vpcProps: defaults.DefaultPublicPrivateVpcProps(),
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail multiple failures message', () => {
    const stack = new core_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n' +
        'Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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