"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./private/cross-region-support-stack");
const full_action_descriptor_1 = require("./private/full-action-descriptor");
const rich_action_1 = require("./private/rich-action");
const stage_1 = require("./private/stage");
const validation_1 = require("./private/validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.pipelineArn,
        };
    }
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    notifyOnExecutionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUCCEEDED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUPERSEDED,
            ],
        });
    }
    notifyOnAnyStageStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_SUCCEEDED,
            ],
        });
    }
    notifyOnAnyActionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_SUCCEEDED,
            ],
        });
    }
    notifyOnAnyManualApprovalStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_FAILED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_NEEDED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_SUCCEEDED,
            ],
        });
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * import * as codecommit from '@aws-cdk/aws-codecommit';
 *
 * const pipeline = new codepipeline.Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ stageName: 'Source' });
 *
 * // add a source action to the stage
 * declare const repo: codecommit.Repository;
 * declare const sourceArtifact: codepipeline.Artifact;
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   output: sourceArtifact,
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_PipelineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // @deprecated(v2): switch to default false
        this.crossAccountKeys = (_b = props.crossAccountKeys) !== null && _b !== void 0 ? _b : true;
        this.enableKeyRotation = props.enableKeyRotation;
        // Cross account keys must be set for key rotation to be enabled
        if (this.enableKeyRotation && !this.crossAccountKeys) {
            throw new Error("Setting 'enableKeyRotation' to true also requires 'crossAccountKeys' to be enabled");
        }
        this.reuseCrossRegionSupportStacks = (_c = props.reuseCrossRegionSupportStacks) !== null && _c !== void 0 ? _c : true;
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            let encryptionKey;
            if (this.crossAccountKeys) {
                encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                    // remove the key - there is a grace period of a few days before it's gone for good,
                    // that should be enough for any emergency access to the bucket artifacts
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                    enableKeyRotation: this.enableKeyRotation,
                });
                // add an alias to make finding the key in the console easier
                new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                    aliasName: this.generateNameForDefaultBucketKeyAlias(),
                    targetKey: encryptionKey,
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                });
            }
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: encryptionKey ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
                enforceSSL: true,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        this.codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.any({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.any({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.any({ produce: () => this.renderStages() }),
            disableInboundStageTransitions: core_1.Lazy.any({ produce: () => this.renderDisabledTransitions() }, { omitEmptyArray: true }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        this.codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(this.codePipeline.ref);
        this.pipelineVersion = this.codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).splitArn(pipelineArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_StageOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addStage);
            }
            throw error;
        }
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Access one of the pipeline's stages by stage name
     */
    stage(stageName) {
        for (const stage of this._stages) {
            if (stage.stageName === stageName) {
                return stage;
            }
        }
        throw new Error(`Pipeline does not contain a stage named '${stageName}'. Available stages: ${this._stages.map(s => s.stageName).join(', ')}`);
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        const richAction = new rich_action_1.RichAction(action, this);
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(richAction);
        // get the role for the given action, handling if it's cross-account
        const actionRole = this.getRoleForAction(stage, richAction, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(richAction.actionProperties.variablesNamespace);
        // bind the Action (type h4x)
        const actionConfig = richAction.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            // must be 'action', not 'richAction',
            // as those are returned by the IStage.actions property,
            // and it's important customers of Pipeline get the same instance
            // back as they added to the pipeline
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        if (!action.isCrossRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // The action has a specific region,
        // require the pipeline to have a known region as well.
        this.requireRegion();
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        const crossRegionSupport = this.obtainCrossRegionSupportFor(action);
        // the stack containing the replication bucket must be deployed before the pipeline
        core_1.Stack.of(this).addDependency(crossRegionSupport.stack);
        // The Pipeline role must be able to replicate to that bucket
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: action.effectiveRegion,
        };
    }
    /**
     * Get or create the cross-region support construct for the given action
     */
    obtainCrossRegionSupportFor(action) {
        // this method is never called for non cross-region actions
        const actionRegion = action.effectiveRegion;
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            const otherStack = action.resourceStack;
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        return crossRegionSupport;
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id, {
                    createKmsKey: this.crossAccountKeys,
                    enableKeyRotation: this.enableKeyRotation,
                });
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.supportScope();
        const supportStackId = `cross-region-stack-${this.reuseCrossRegionSupportStacks ? pipelineAccount : pipelineStack.stackName}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
                createKmsKey: this.crossAccountKeys,
                enableKeyRotation: this.enableKeyRotation,
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = core_1.Names.uniqueId(this);
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        const grant = actionRole === null || actionRole === void 0 ? void 0 : actionRole.grantAssumeRole(this.role);
        grant === null || grant === void 0 ? void 0 : grant.applyBefore(this.codePipeline);
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        // (otherwise we can't configure cross-account trust policies)
        if (action.isCrossAccount) {
            const artifactBucket = this.ensureReplicationResourcesExistFor(action).artifactBucket;
            if (!artifactBucket.encryptionKey) {
                throw new Error(`Artifact Bucket must have a KMS Key to add cross-account action '${action.actionProperties.actionName}' ` +
                    `(pipeline account: '${renderEnvDimension(this.env.account)}', action account: '${renderEnvDimension(action.effectiveAccount)}'). ` +
                    'Create Pipeline with \'crossAccountKeys: true\' (or pass an existing Bucket with a key)');
            }
        }
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${core_1.Names.uniqueId(this)}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const targetAccount = action.actionProperties.resource
            ? action.actionProperties.resource.env.account
            : action.actionProperties.account;
        if (targetAccount === undefined) {
            // if the account of the Action is not specified,
            // then it defaults to the same account the pipeline itself is in
            return undefined;
        }
        // check whether the action's account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            if (core_1.Token.isUnresolved(this.env.account)) {
                // the pipeline is also env-agnostic, so that's fine
                return undefined;
            }
            else {
                throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
            }
        }
        // At this point, we know that the action's account is a static string.
        // In this case, the pipeline's account must also be a static string.
        if (core_1.Token.isUnresolved(this.env.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        // at this point, we know that both the Pipeline's account,
        // and the action-backing resource's account are static strings
        // if they are identical - nothing to do (the action is not cross-account)
        if (this.env.account === targetAccount) {
            return undefined;
        }
        // at this point, we know that the action is certainly cross-account,
        // so we need to return a Stack in its account to create the helper Role in
        const candidateActionResourceStack = action.actionProperties.resource
            ? core_1.Stack.of(action.actionProperties.resource)
            : undefined;
        if ((candidateActionResourceStack === null || candidateActionResourceStack === void 0 ? void 0 : candidateActionResourceStack.account) === targetAccount) {
            // we always use the "latest" action-backing resource's Stack for this account,
            // even if a different one was used earlier
            this._crossAccountSupport[targetAccount] = candidateActionResourceStack;
            return candidateActionResourceStack;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.supportScope();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                const actionRegion = action.actionProperties.resource
                    ? action.actionProperties.resource.env.region
                    : action.actionProperties.region;
                const pipelineStack = core_1.Stack.of(this);
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: actionRegion !== null && actionRegion !== void 0 ? actionRegion : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    renderDisabledTransitions() {
        return this._stages
            .filter(stage => !stage.transitionToEnabled)
            .map(stage => ({
            reason: stage.transitionDisabledReason,
            stageName: stage.stageName,
        }));
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    supportScope() {
        const scope = core_1.Stage.of(this);
        if (!scope) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK App or Stage');
        }
        return scope;
    }
}
exports.Pipeline = Pipeline;
_a = JSII_RTTI_SYMBOL_1;
Pipeline[_a] = { fqn: "@aws-cdk/aws-codepipeline.Pipeline", version: "1.159.0" };
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
/**
 * Render an env dimension without showing the ugly stringified tokens
 */
function renderEnvDimension(s) {
    return core_1.Token.isUnresolved(s) ? '(current)' : s;
}
//# sourceMappingURL=data:application/json;base64,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