"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const path = require("path");
class LambdaPowerTuner extends cdk.Construct {
    constructor(scope, id, config) {
        var _a, _b, _c;
        super(scope, id);
        let powerValues = (_a = config.powerValues) !== null && _a !== void 0 ? _a : [128, 256, 512, 1024, 1536, 3008];
        const REGION = (_b = process.env.CDK_DEFAULT_REGION) !== null && _b !== void 0 ? _b : 'us-east-1';
        let shared_env = {
            defaultPowerValues: powerValues.join(','),
            minRAM: '128',
            baseCosts: '{"ap-east-1":2.865e-7,"af-south-1":2.763e-7,"me-south-1":2.583e-7,"eu-south-1":2.440e-7,"default":2.083e-7}',
            visualizationURL: (_c = config.visualizationURL) !== null && _c !== void 0 ? _c : 'https://lambda-power-tuning.show/'
        };
        // Initializer
        let initializer = this.createLambda(this, 'initializer', 'initializer.handler', shared_env);
        let lambdaConfigPermissions = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [config.lambdaResource],
            actions: ['lambda:GetAlias',
                'lambda:GetFunctionConfiguration',
                'lambda:PublishVersion',
                'lambda:UpdateFunctionConfiguration',
                'lambda:CreateAlias',
                'lambda:UpdateAlias']
        });
        initializer.addToRolePolicy(lambdaConfigPermissions);
        // Executor
        let executor = this.createLambda(this, 'executor', 'executor.handler', shared_env);
        executor.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [config.lambdaResource],
            actions: ['lambda:InvokeFunction']
        }));
        // Cleaner
        let cleaner = this.createLambda(this, 'cleaner', 'cleaner.handler', shared_env);
        cleaner.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [config.lambdaResource],
            actions: ['lambda:GetAlias',
                'lambda:DeleteAlias',
                'lambda:DeleteFunction']
        }));
        // Analyzer
        let analyzer = this.createLambda(this, 'analyzer', 'analyzer.handler', shared_env, 10);
        // Optimizer
        let optimizer = this.createLambda(this, 'optimizer', 'optimizer.handler', shared_env);
        optimizer.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [config.lambdaResource],
            actions: ['lambda:GetAlias',
                'lambda:PublishVersion',
                'lambda:UpdateFunctionConfiguration',
                'lambda:CreateAlias',
                'lambda:UpdateAlias'
            ]
        }));
        /**
         * State Machine
         */
        let statemachineRole = new iam.Role(this, 'statemachineRole', {
            assumedBy: new iam.ServicePrincipal(`states.${REGION}.amazonaws.com`),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaRole'),
            ]
        });
        const CleanUpOnError = new sfn.Task(this, 'CleanUpOnError', {
            task: new tasks.RunLambdaTask(cleaner)
        });
        const branching = new sfn.Map(this, 'Branching', {
            itemsPath: sfn.Data.stringAt("$.powerValues"),
            resultPath: "$.stats",
            parameters: {
                "value.$": "$$.Map.Item.Value",
                "lambdaARN.$": "$.lambdaARN",
                "num.$": "$.num",
                "payload.$": "$.payload",
                "parallelInvocation.$": "$.parallelInvocation"
            },
            maxConcurrency: 0,
        }).addCatch(CleanUpOnError, {
            resultPath: "$.error",
        });
        branching.iterator(new sfn.Task(this, 'Iterator', {
            task: new tasks.RunLambdaTask(executor)
        }).addRetry({
            maxAttempts: 2,
            interval: cdk.Duration.seconds(3)
        }));
        const initializerTask = new sfn.Task(this, 'Initializer', {
            task: new tasks.RunLambdaTask(initializer),
            resultPath: "$.powerValues"
        }).addCatch(CleanUpOnError, {
            resultPath: "$.error"
        });
        const cleanupPowerValues = new sfn.Pass(this, 'cleanup $.powerValues', {
            inputPath: '$.powerValues.Payload',
            resultPath: '$.powerValues',
        });
        const cleanupStats = new sfn.Pass(this, 'cleanup $.stats array', {
            inputPath: '$.stats[*].Payload',
            resultPath: '$.stats',
        });
        const cleanerTask = new sfn.Task(this, 'Cleaner', {
            task: new tasks.RunLambdaTask(cleaner),
            resultPath: "$.cleaner"
        });
        const analyzerTask = new sfn.Task(this, 'Analyzer', {
            task: new tasks.RunLambdaTask(analyzer),
            resultPath: "$.analysis"
        });
        const cleanupAnalyzer = new sfn.Pass(this, 'cleanup $.analysis', {
            inputPath: '$.analysis.Payload',
            resultPath: '$.analysis',
        });
        const optimizerTask = new sfn.Task(this, 'Optimizer', {
            task: new tasks.RunLambdaTask(optimizer),
            resultPath: '$.optimizer',
            outputPath: "$.analysis"
        });
        //Step function definition
        const definition = sfn.Chain
            .start(initializerTask)
            .next(cleanupPowerValues)
            .next(branching)
            .next(cleanupStats)
            .next(cleanerTask)
            .next(analyzerTask)
            .next(cleanupAnalyzer)
            .next(optimizerTask);
        const stateMachine = new sfn.StateMachine(this, 'LambdaPowerTuner', {
            definition,
            role: statemachineRole
        });
        new cdk.CfnOutput(this, 'StateMachineARN', {
            value: stateMachine.stateMachineArn
        });
    }
    /**
     * All the lambdas have the same config, so this method saves typing
     * @param scope
     * @param id
     * @param handler
     * @param env
     * @param timeout
     */
    createLambda(scope, id, handler, env, timeout) {
        let role = new iam.Role(scope, `${id}LambdaExecuteRole`, {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AWSLambdaExecute'),
            ],
        });
        return new lambda.Function(scope, id, {
            runtime: lambda.Runtime.NODEJS_12_X,
            role: role,
            timeout: cdk.Duration.seconds(timeout !== null && timeout !== void 0 ? timeout : 300),
            memorySize: 128,
            code: lambda.Code.asset(path.join(__dirname, '../../powertuner_clone/lambda')),
            handler: handler,
            environment: env
        });
    }
}
exports.LambdaPowerTuner = LambdaPowerTuner;
//# sourceMappingURL=data:application/json;base64,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