"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const cdk_lambda_powertuner_construct_1 = require("../lib/cdk-lambda-powertuner/cdk-lambda-powertuner-construct");
test('initializer Lambda Created', () => {
    const app = new cdk.App();
    // WHEN
    const stack = new cdk.Stack(app, 'MyTestStack');
    new cdk_lambda_powertuner_construct_1.LambdaPowerTuner(stack, 'powerTuner', {
        lambdaResource: 'arn:1234:1234:1234:1234:1234'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::Lambda::Function", {
        "Handler": "initializer.handler",
        "Runtime": "nodejs12.x",
        "Environment": {
            "Variables": {
                "defaultPowerValues": "128,256,512,1024,1536,3008",
                "minRAM": "128",
                "baseCosts": "{\"ap-east-1\":2.865e-7,\"af-south-1\":2.763e-7,\"me-south-1\":2.583e-7,\"eu-south-1\":2.440e-7,\"default\":2.083e-7}",
                "visualizationURL": "https://lambda-power-tuning.show/"
            }
        },
        "MemorySize": 128,
        "Timeout": 300
    }));
});
test('executor Lambda Created', () => {
    const app = new cdk.App();
    // WHEN
    const stack = new cdk.Stack(app, 'MyTestStack');
    new cdk_lambda_powertuner_construct_1.LambdaPowerTuner(stack, 'powerTuner', {
        lambdaResource: 'arn:1234:1234:1234:1234:1234'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::Lambda::Function", {
        "Handler": "executor.handler",
        "Runtime": "nodejs12.x",
        "Environment": {
            "Variables": {
                "defaultPowerValues": "128,256,512,1024,1536,3008",
                "minRAM": "128",
                "baseCosts": "{\"ap-east-1\":2.865e-7,\"af-south-1\":2.763e-7,\"me-south-1\":2.583e-7,\"eu-south-1\":2.440e-7,\"default\":2.083e-7}",
                "visualizationURL": "https://lambda-power-tuning.show/"
            }
        },
        "MemorySize": 128,
        "Timeout": 300
    }));
});
test('cleaner Lambda Created', () => {
    const app = new cdk.App();
    // WHEN
    const stack = new cdk.Stack(app, 'MyTestStack');
    new cdk_lambda_powertuner_construct_1.LambdaPowerTuner(stack, 'powerTuner', {
        lambdaResource: 'arn:1234:1234:1234:1234:1234'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::Lambda::Function", {
        "Handler": "cleaner.handler",
        "Runtime": "nodejs12.x",
        "Environment": {
            "Variables": {
                "defaultPowerValues": "128,256,512,1024,1536,3008",
                "minRAM": "128",
                "baseCosts": "{\"ap-east-1\":2.865e-7,\"af-south-1\":2.763e-7,\"me-south-1\":2.583e-7,\"eu-south-1\":2.440e-7,\"default\":2.083e-7}",
                "visualizationURL": "https://lambda-power-tuning.show/"
            }
        },
        "MemorySize": 128,
        "Timeout": 300
    }));
});
test('analyzer Lambda Created', () => {
    const app = new cdk.App();
    // WHEN
    const stack = new cdk.Stack(app, 'MyTestStack');
    new cdk_lambda_powertuner_construct_1.LambdaPowerTuner(stack, 'powerTuner', {
        lambdaResource: 'arn:1234:1234:1234:1234:1234'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::Lambda::Function", {
        "Handler": "analyzer.handler",
        "Runtime": "nodejs12.x",
        "Environment": {
            "Variables": {
                "defaultPowerValues": "128,256,512,1024,1536,3008",
                "minRAM": "128",
                "baseCosts": "{\"ap-east-1\":2.865e-7,\"af-south-1\":2.763e-7,\"me-south-1\":2.583e-7,\"eu-south-1\":2.440e-7,\"default\":2.083e-7}",
                "visualizationURL": "https://lambda-power-tuning.show/"
            }
        },
        "MemorySize": 128,
        "Timeout": 10
    }));
});
test('optimizer Lambda Created', () => {
    const app = new cdk.App();
    // WHEN
    const stack = new cdk.Stack(app, 'MyTestStack');
    new cdk_lambda_powertuner_construct_1.LambdaPowerTuner(stack, 'powerTuner', {
        lambdaResource: 'arn:1234:1234:1234:1234:1234'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::Lambda::Function", {
        "Handler": "optimizer.handler",
        "Runtime": "nodejs12.x",
        "Environment": {
            "Variables": {
                "defaultPowerValues": "128,256,512,1024,1536,3008",
                "minRAM": "128",
                "baseCosts": "{\"ap-east-1\":2.865e-7,\"af-south-1\":2.763e-7,\"me-south-1\":2.583e-7,\"eu-south-1\":2.440e-7,\"default\":2.083e-7}",
                "visualizationURL": "https://lambda-power-tuning.show/"
            }
        },
        "MemorySize": 128,
        "Timeout": 300
    }));
});
test('State Machine Created', () => {
    const app = new cdk.App();
    // WHEN
    const stack = new cdk.Stack(app, 'MyTestStack');
    new cdk_lambda_powertuner_construct_1.LambdaPowerTuner(stack, 'powerTuner', {
        lambdaResource: 'arn:1234:1234:1234:1234:1234'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::StepFunctions::StateMachine", {
        "DefinitionString": {
            "Fn::Join": [
                "",
                [
                    "{\"StartAt\":\"Initializer\",\"States\":{\"Initializer\":{\"Next\":\"cleanup $.powerValues\",\"Catch\":[{\"ErrorEquals\":[\"States.ALL\"],\"ResultPath\":\"$.error\",\"Next\":\"CleanUpOnError\"}],\"Parameters\":{\"FunctionName\":\"",
                    {},
                    "\",\"Payload.$\":\"$\"},\"Type\":\"Task\",\"Resource\":\"arn:",
                    {},
                    ":states:::lambda:invoke\",\"ResultPath\":\"$.powerValues\"},\"cleanup $.powerValues\":{\"Type\":\"Pass\",\"ResultPath\":\"$.powerValues\",\"InputPath\":\"$.powerValues.Payload\",\"Next\":\"Branching\"},\"Branching\":{\"Type\":\"Map\",\"ResultPath\":\"$.stats\",\"Next\":\"cleanup $.stats array\",\"Parameters\":{\"value.$\":\"$$.Map.Item.Value\",\"lambdaARN.$\":\"$.lambdaARN\",\"num.$\":\"$.num\",\"payload.$\":\"$.payload\",\"parallelInvocation.$\":\"$.parallelInvocation\"},\"Catch\":[{\"ErrorEquals\":[\"States.ALL\"],\"ResultPath\":\"$.error\",\"Next\":\"CleanUpOnError\"}],\"Iterator\":{\"StartAt\":\"Iterator\",\"States\":{\"Iterator\":{\"End\":true,\"Retry\":[{\"ErrorEquals\":[\"States.ALL\"],\"IntervalSeconds\":3,\"MaxAttempts\":2}],\"Parameters\":{\"FunctionName\":\"",
                    {},
                    "\",\"Payload.$\":\"$\"},\"Type\":\"Task\",\"Resource\":\"arn:",
                    {},
                    ":states:::lambda:invoke\"}}},\"ItemsPath\":\"$.powerValues\",\"MaxConcurrency\":0},\"cleanup $.stats array\":{\"Type\":\"Pass\",\"ResultPath\":\"$.stats\",\"InputPath\":\"$.stats[*].Payload\",\"Next\":\"Cleaner\"},\"Cleaner\":{\"Next\":\"Analyzer\",\"Parameters\":{\"FunctionName\":\"",
                    {},
                    "\",\"Payload.$\":\"$\"},\"Type\":\"Task\",\"Resource\":\"arn:",
                    {},
                    ":states:::lambda:invoke\",\"ResultPath\":\"$.cleaner\"},\"Analyzer\":{\"Next\":\"cleanup $.analysis\",\"Parameters\":{\"FunctionName\":\"",
                    {},
                    "\",\"Payload.$\":\"$\"},\"Type\":\"Task\",\"Resource\":\"arn:",
                    {},
                    ":states:::lambda:invoke\",\"ResultPath\":\"$.analysis\"},\"cleanup $.analysis\":{\"Type\":\"Pass\",\"ResultPath\":\"$.analysis\",\"InputPath\":\"$.analysis.Payload\",\"Next\":\"Optimizer\"},\"Optimizer\":{\"End\":true,\"Parameters\":{\"FunctionName\":\"",
                    {},
                    "\",\"Payload.$\":\"$\"},\"OutputPath\":\"$.analysis\",\"Type\":\"Task\",\"Resource\":\"arn:",
                    {},
                    ":states:::lambda:invoke\",\"ResultPath\":\"$.optimizer\"},\"CleanUpOnError\":{\"End\":true,\"Parameters\":{\"FunctionName\":\"",
                    {},
                    "\",\"Payload.$\":\"$\"},\"Type\":\"Task\",\"Resource\":\"arn:",
                    {},
                    ":states:::lambda:invoke\"}}}"
                ]
            ]
        }
    }));
});
//# sourceMappingURL=data:application/json;base64,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