# Licensed to the Software Freedom Conservancy (SFC) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

from datetime import timedelta
from typing import Optional, Union

from appium.options.common.supports_capabilities import SupportsCapabilities

WDA_STARTUP_RETRY_INTERVAL = 'wdaStartupRetryInterval'


class WdaStartupRetryIntervalOption(SupportsCapabilities):
    @property
    def wda_startup_retry_interval(self) -> Optional[timedelta]:
        """
        Interval to wait between tries to build and launch WebDriverAgent.
        """
        value = self.get_capability(WDA_STARTUP_RETRY_INTERVAL)
        return None if value is None else timedelta(milliseconds=value)

    @wda_startup_retry_interval.setter
    def wda_startup_retry_interval(self, value: Union[timedelta, int]) -> None:
        """
        Time interval to wait between tries to build and launch WebDriverAgent.
        Defaults to 10000ms.
        """
        self.set_capability(
            WDA_STARTUP_RETRY_INTERVAL, value.microseconds // 1000 if isinstance(value, timedelta) else value
        )
