"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const logger = log4js_1.getLogger();
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sampleCode: false,
            tsconfig: {
                ...projectOptions.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    ...projectOptions.tsconfig?.compilerOptions,
                },
            },
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        logger.debug(`specDir = "${this.specDir}"`);
        logger.debug(`specFileName = "${this.specFileName}"`);
        logger.debug(`generatedCodeDir = "${this.generatedCodeDir}"`);
        logger.debug(`forceGenerateCodeAndDocs = ${this.forceGenerateCodeAndDocs}`);
        logger.debug(`apiSrcDir = "${this.apiSrcDir}"`);
        // Allow json files to be imported (for importing the parsed spec)
        this.tsconfig?.addInclude(`${this.srcdir}/**/*.json`);
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib", "cdk-nag");
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.srcdir, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Delete the lib directory prior to compilation to ensure latest parsed spec json file is copied
        this.preCompileTask.exec(`rm -rf ${this.libdir}`);
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.TYPESCRIPT);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.TYPESCRIPT,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.package.packageName,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: options.defaultReleaseBranch,
                packageManager: options.packageManager,
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedTypescriptClient = this.generatedClients[languages_1.ClientLanguage.TYPESCRIPT];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            apiSrcDir: path.join(this.srcdir, this.apiSrcDir),
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleFile(this, path.join(this.srcdir, "index.ts"), {
            contents: `export * from "./${this.apiSrcDir}";`,
        });
        new projen_1.SampleDir(this, path.join(this.srcdir, this.apiSrcDir), {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than typescript project options.
     */
    preConstruct(options) {
        return options;
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.11.0" };
//# sourceMappingURL=data:application/json;base64,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